<?php

namespace Drupal\dify_widget_vanilla\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for generating dynamic CSS for Dify Widget.
 */
class DifyWidgetCssController extends ControllerBase {

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new DifyWidgetCssController object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(StateInterface $state) {
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('state')
    );
  }

  /**
   * Generates CSS with custom colors.
   *
   * @param string $hash
   *   The hash identifying the color configuration.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The CSS response.
   */
  public function generateCss($hash) {
    // Validate hash format.
    if (!preg_match('/^[a-f0-9]{32}$/', $hash)) {
      throw new NotFoundHttpException();
    }

    // Get custom colors from state.
    $custom_colors = $this->state->get("dify_widget_colors_{$hash}");
    if (!$custom_colors || !is_array($custom_colors)) {
      throw new NotFoundHttpException();
    }

    // Generate CSS with validated colors.
    $css = ":root {\n";
    foreach ($custom_colors as $key => $color) {
      if ($this->isValidCssColor($color)) {
        $safe_key = preg_replace('/[^a-z-]/', '', $key);
        $css .= "  --dify-{$safe_key}-custom: {$color};\n";
      }
    }
    $css .= "}\n";

    // Return CSS response with security headers.
    return new Response($css, 200, [
      'Content-Type' => 'text/css',
      'Cache-Control' => 'public, max-age=3600',
      'X-Content-Type-Options' => 'nosniff',
    ]);
  }

  /**
   * Validates if a string is a valid CSS color.
   *
   * @param string $color
   *   The color value to validate.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  private function isValidCssColor($color) {
    if (!is_string($color)) {
      return FALSE;
    }

    $color = trim($color);

    // Allow hex colors, rgb/rgba, hsl/hsla, and CSS color names.
    $pattern = '/^(#[0-9a-fA-F]{3,6}|rgb\(\s*\d+\s*,\s*\d+\s*,\s*\d+\s*\)|rgba\(\s*\d+\s*,\s*\d+\s*,\s*\d+\s*,\s*[0-9.]+\s*\)|hsl\(\s*\d+\s*,\s*\d+%\s*,\s*\d+%\s*\)|hsla\(\s*\d+\s*,\s*\d+%\s*,\s*\d+%\s*,\s*[0-9.]+\s*\)|transparent|inherit|initial|unset|[a-zA-Z]+)$/';

    return preg_match($pattern, $color) === 1;
  }

}
