<?php

namespace Drupal\dify_widget_vanilla\Controller;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\dify_widget_vanilla\Service\MarkdownService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for markdown rendering.
 */
class MarkdownController implements ContainerInjectionInterface {

  /**
   * The markdown service.
   *
   * @var \Drupal\dify_widget_vanilla\Service\MarkdownService
   */
  protected $markdownService;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructs a new MarkdownController object.
   *
   * @param \Drupal\dify_widget_vanilla\Service\MarkdownService $markdown_service
   *   The markdown service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(MarkdownService $markdown_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->markdownService = $markdown_service;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dify_widget_vanilla.markdown_service'),
      $container->get('logger.factory')
    );
  }

  /**
   * Renders markdown content to HTML.
   */
  public function render(Request $request): JsonResponse {
    $data = json_decode($request->getContent(), TRUE);
    $markdown = $data['markdown'] ?? '';
    $is_complete = $data['is_complete'] ?? FALSE;

    if (empty($markdown)) {
      return new JsonResponse(['html' => '']);
    }

    try {
      $html = $this->markdownService->toHtmlStreaming($markdown, $is_complete);
      return new JsonResponse(['html' => $html]);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('dify_widget_vanilla')->error('Markdown rendering error: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse(['html' => htmlspecialchars($markdown, ENT_QUOTES | ENT_HTML5, 'UTF-8')]);
    }
  }

}
