<?php

namespace Drupal\digital_signage_framework;

use Drupal\Core\State\StateInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Provides middleware to implement Digital Signage based language fallback.
 */
class Middleware implements HttpKernelInterface {

  /**
   * Flags if the middleware is active, can be disabled with ::disable().
   *
   * @var bool
   */
  protected static bool $active = TRUE;

  /**
   * The decorated kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface
   */
  protected HttpKernelInterface $httpKernel;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * Constructs a Digital Signage Middleware object.
   *
   * @param \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel
   *   The decorated kernel.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state.
   */
  public function __construct(HttpKernelInterface $http_kernel, StateInterface $state) {
    $this->httpKernel = $http_kernel;
    $this->state = $state;
  }

  /**
   * Turns off this middleware for the current request.
   */
  public static function disable(): void {
    self::$active = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function handle(Request $request, $type = HttpKernelInterface::MAIN_REQUEST, $catch = TRUE): Response {
    $path = $request->getPathInfo();
    $dir = '/sites/default/files/languages/';
    if (self::$active && str_starts_with($path, $dir) && !file_exists(DRUPAL_ROOT . $path)) {
      $path = str_replace($dir, '', $path);
      $langcode = substr($path, 0, strpos($path, '_'));
      $locale_javascripts = $this->state->get('locale.translation.javascript', []);
      if (isset($locale_javascripts[$langcode])) {
        return new Response(file_get_contents(DRUPAL_ROOT . $dir . $langcode . '_' . $locale_javascripts[$langcode] . '.js'));
      }
    }
    return $this->httpKernel->handle($request, $type, $catch);
  }

}
