<?php

namespace Drupal\Tests\disable_account_emails\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Disable Account Emails module functionality.
 *
 * Verifies that the module correctly integrates with the account settings
 * form, saves configuration properly, and alters email sending behavior
 * as expected.
 *
 * @group disable_account_emails
 */
class DisableAccountEmailsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'disable_account_emails',
  ];

  /**
   * A user with permission to administer account settings.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user without permission to administer account settings.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user with account settings permission.
    $this->adminUser = $this->drupalCreateUser(['administer account settings']);

    // Create normal user without account settings permission.
    $this->normalUser = $this->drupalCreateUser();
  }

  /**
   * Tests that the module settings are accessible to authorized users.
   */
  public function testSettingsAccess(): void {
    // Login as admin user.
    $this->drupalLogin($this->adminUser);

    // Access the account settings page.
    $this->drupalGet('admin/config/people/accounts');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Disable Account Emails');

    // Check that all email types are listed.
    $email_types = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'register_activate',
      'status_blocked',
      'cancel_confirm',
      'status_deleted',
      'password_reset',
    ];

    foreach ($email_types as $key) {
      $field_name = 'disable_account_emails_' . $key;
      $this->assertSession()->fieldExists($field_name);
    }

    // Logout admin user.
    $this->drupalLogout();

    // Login as normal user.
    $this->drupalLogin($this->normalUser);

    // Try to access the account settings page.
    $this->drupalGet('admin/config/people/accounts');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the module settings form works correctly.
   */
  public function testSettingsForm(): void {
    // Login as admin user.
    $this->drupalLogin($this->adminUser);

    // Access the account settings page.
    $this->drupalGet('admin/config/people/accounts');

    // Submit the form with some emails disabled.
    $edit = [
      'disable_account_emails_register_no_approval_required' => TRUE,
      'disable_account_emails_password_reset' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Check that the configuration was saved correctly.
    $config = $this->config('disable_account_emails.settings');
    $disabled_emails = $config->get('disabled_emails');
    $this->assertTrue($disabled_emails['register_no_approval_required']);
    $this->assertTrue($disabled_emails['password_reset']);
    $this->assertFalse($disabled_emails['register_admin_created']);

    // Test that unchecking works by submitting again with different values.
    $edit = [
      'disable_account_emails_register_no_approval_required' => FALSE,
      'disable_account_emails_password_reset' => TRUE,
      'disable_account_emails_register_admin_created' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Check that the configuration was updated correctly.
    $config = $this->config('disable_account_emails.settings');
    $disabled_emails = $config->get('disabled_emails');
    $this->assertFalse($disabled_emails['register_no_approval_required']);
    $this->assertTrue($disabled_emails['password_reset']);
    $this->assertTrue($disabled_emails['register_admin_created']);
  }

  /**
   * Tests that the mail alter hook works correctly.
   */
  public function testMailAlter(): void {
    // Set up configuration to disable a specific email.
    $config = \Drupal::configFactory()->getEditable('disable_account_emails.settings');
    $config->set('disabled_emails.register_no_approval_required', TRUE);
    $config->save();

    // Create a mock message.
    $message = [
      'module' => 'user',
      'key' => 'register_no_approval_required',
      'send' => TRUE,
    ];

    // Call the mail alter hook.
    disable_account_emails_mail_alter($message);

    // Check that the email was disabled.
    $this->assertFalse($message['send']);

    // Test with an email that is not disabled.
    $message = [
      'module' => 'user',
      'key' => 'password_reset',
      'send' => TRUE,
    ];

    disable_account_emails_mail_alter($message);

    // Check that the email was not disabled.
    $this->assertTrue($message['send']);

    // Test with a non-user module email.
    $message = [
      'module' => 'other_module',
      'key' => 'register_no_approval_required',
      'send' => TRUE,
    ];

    disable_account_emails_mail_alter($message);

    // Check that the email was not affected.
    $this->assertTrue($message['send']);
  }

  /**
   * Tests that the module uninstall removes the configuration.
   */
  public function testUninstall(): void {
    // Install the module and set some configuration.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/people/accounts');
    $edit = [
      'disable_account_emails_register_no_approval_required' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify the configuration exists.
    $config = $this->config('disable_account_emails.settings');
    $this->assertNotEmpty($config->get('disabled_emails'));

    // Uninstall the module.
    \Drupal::service('module_installer')->uninstall(['disable_account_emails']);

    // Verify the configuration has been removed.
    $config = $this->config('disable_account_emails.settings');
    $this->assertEmpty($config->getRawData());
  }

  /**
   * Tests that default configuration is correct after installation.
   */
  public function testDefaultConfiguration(): void {
    // Load the configuration.
    $config = $this->config('disable_account_emails.settings');
    $disabled_emails = $config->get('disabled_emails');

    // Verify that disabled_emails exists and is an array.
    $this->assertIsArray($disabled_emails, 'disabled_emails should be an array');

    // Get all expected email types.
    $expected_email_types = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'register_activate',
      'status_blocked',
      'cancel_confirm',
      'status_deleted',
      'password_reset',
    ];

    // Verify all email types are present.
    foreach ($expected_email_types as $key) {
      $this->assertArrayHasKey($key, $disabled_emails, "Email type '$key' should exist in configuration");
    }

    // Verify all email types are set to FALSE by default (all emails enabled).
    foreach ($disabled_emails as $key => $value) {
      $this->assertFalse($value, "Email type '$key' should be FALSE (enabled) by default");
    }

    // Verify there are exactly 9 email types.
    $this->assertCount(9, $disabled_emails, 'There should be exactly 9 email types configured');
  }

  /**
   * Tests behavior when all email types are disabled.
   */
  public function testAllEmailTypesDisabled(): void {
    // Set up configuration to disable all emails.
    $config = \Drupal::configFactory()->getEditable('disable_account_emails.settings');
    $all_disabled = [
      'register_admin_created' => TRUE,
      'register_pending_approval' => TRUE,
      'register_pending_approval_admin' => TRUE,
      'register_no_approval_required' => TRUE,
      'register_activate' => TRUE,
      'status_blocked' => TRUE,
      'cancel_confirm' => TRUE,
      'status_deleted' => TRUE,
      'password_reset' => TRUE,
    ];
    $config->set('disabled_emails', $all_disabled);
    $config->save();

    // Test each email type to ensure it's disabled.
    foreach (array_keys($all_disabled) as $email_key) {
      $message = [
        'module' => 'user',
        'key' => $email_key,
        'send' => TRUE,
      ];

      disable_account_emails_mail_alter($message);

      $this->assertFalse($message['send'], "Email '$email_key' should be disabled");
    }

    // Verify that a non-user module email is still not affected.
    $message = [
      'module' => 'contact',
      'key' => 'page_mail',
      'send' => TRUE,
    ];

    disable_account_emails_mail_alter($message);
    $this->assertTrue($message['send'], 'Non-user module emails should not be affected');

    // Verify the form displays all checkboxes as checked.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/people/accounts');

    foreach (array_keys($all_disabled) as $email_key) {
      $field_name = 'disable_account_emails_' . $email_key;
      $this->assertSession()->checkboxChecked($field_name);
    }
  }

  /**
   * Tests the structure of the form integration.
   */
  public function testFormStructure(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/people/accounts');

    // Verify the details element exists and is properly structured.
    $page = $this->getSession()->getPage();

    // Check that the details element exists.
    $details = $page->find('css', 'details');
    $this->assertNotNull($details, 'Details element should exist on the page');

    // Verify the details element has the correct title.
    $summary = $page->find('css', 'details summary');
    $this->assertNotNull($summary, 'Details summary should exist');
    $this->assertStringContainsString('Disable Account Emails', $summary->getText());

    // Verify the details element is open by default.
    $this->assertTrue($details->hasAttribute('open'), 'Details element should be open by default');

    // Verify all checkboxes use the correct #parents structure.
    $email_types = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'register_activate',
      'status_blocked',
      'cancel_confirm',
      'status_deleted',
      'password_reset',
    ];

    foreach ($email_types as $key) {
      $field_name = 'disable_account_emails_' . $key;
      $checkbox = $page->findField($field_name);
      $this->assertNotNull($checkbox, "Checkbox for '$key' should exist");
      $this->assertEquals('checkbox', $checkbox->getAttribute('type'), "Field '$field_name' should be a checkbox");

      // Verify the field name follows the expected pattern.
      $name_attribute = $checkbox->getAttribute('name');
      $this->assertEquals('disable_account_emails[' . $key . ']', $name_attribute, "Checkbox name should use parents structure");
    }

    // Verify the description is present.
    $this->assertSession()->pageTextContains('Select which user account emails should be disabled.');
  }

}