<?php

namespace Drupal\Tests\disable_account_emails\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests configuration schema for the Disable Account Emails module.
 *
 * @group disable_account_emails
 */
class ConfigSchemaTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'disable_account_emails',
  ];

  /**
   * Tests that the configuration schema is valid.
   */
  public function testConfigSchemaValidation(): void {
    // Install the module configuration.
    $this->installConfig(['disable_account_emails']);

    // Load the configuration.
    $config = $this->config('disable_account_emails.settings');

    // Verify the configuration exists.
    $this->assertNotNull($config, 'Configuration should exist');

    // Get the typed config manager.
    $typed_config = \Drupal::service('config.typed');

    // Get the schema for our configuration.
    $definition = $typed_config->getDefinition('disable_account_emails.settings');

    // Verify the schema definition exists.
    $this->assertNotEmpty($definition, 'Configuration schema should be defined');

    // Verify the schema type is correct.
    $this->assertEquals('disable_account_emails.settings', $definition['type']);

    // Get the typed configuration data.
    $schema = $typed_config->create($definition, $config->get());

    // Verify the schema validates correctly.
    $this->assertInstanceOf('Drupal\Core\Config\Schema\Mapping', $schema, 'Schema should be a Mapping type');

    // Verify disabled_emails is defined in the schema.
    $elements = $schema->getElements();
    $this->assertArrayHasKey('disabled_emails', $elements, 'disabled_emails should be in schema');

    // Verify disabled_emails is a mapping type.
    $disabled_emails_schema = $elements['disabled_emails'];
    $this->assertInstanceOf('Drupal\Core\Config\Schema\Mapping', $disabled_emails_schema, 'disabled_emails should be a Mapping type');

    // Verify all email type keys are defined in the schema.
    $email_elements = $disabled_emails_schema->getElements();
    $expected_keys = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'register_activate',
      'status_blocked',
      'cancel_confirm',
      'status_deleted',
      'password_reset',
    ];

    foreach ($expected_keys as $key) {
      $this->assertArrayHasKey($key, $email_elements, "Schema should define '$key'");
    }

    // Verify each email type is a boolean.
    foreach ($email_elements as $key => $element) {
      $this->assertInstanceOf('Drupal\Core\Config\Schema\Boolean', $element, "Email type '$key' should be a Boolean in schema");
    }
  }

  /**
   * Tests that invalid configuration values are rejected.
   */
  public function testInvalidConfigurationValues(): void {
    $this->installConfig(['disable_account_emails']);

    // Get the configuration object.
    $config = $this->config('disable_account_emails.settings');

    // Test that the default values are valid.
    $disabled_emails = $config->get('disabled_emails');
    $this->assertIsArray($disabled_emails);

    foreach ($disabled_emails as $key => $value) {
      $this->assertIsBool($value, "Value for '$key' should be boolean");
    }

    // Test setting a valid boolean value.
    $config_editable = \Drupal::configFactory()->getEditable('disable_account_emails.settings');
    $config_editable->set('disabled_emails.register_no_approval_required', TRUE)->save();

    $updated_config = $this->config('disable_account_emails.settings');
    $this->assertTrue($updated_config->get('disabled_emails.register_no_approval_required'));
  }

  /**
   * Tests that configuration can be imported and exported.
   */
  public function testConfigurationImportExport(): void {
    $this->installConfig(['disable_account_emails']);

    // Get the original configuration.
    $original_config = $this->config('disable_account_emails.settings');
    $original_data = $original_config->getRawData();

    // Modify the configuration.
    $config_editable = \Drupal::configFactory()->getEditable('disable_account_emails.settings');
    $config_editable->set('disabled_emails.register_no_approval_required', TRUE);
    $config_editable->set('disabled_emails.password_reset', TRUE);
    $config_editable->save();

    // Export the configuration.
    $exported_data = $config_editable->getRawData();

    // Verify the exported data contains our changes.
    $this->assertTrue($exported_data['disabled_emails']['register_no_approval_required']);
    $this->assertTrue($exported_data['disabled_emails']['password_reset']);

    // Reset to original.
    $config_editable->setData($original_data)->save();

    // Verify reset worked.
    $reset_config = $this->config('disable_account_emails.settings');
    $this->assertEquals($original_data, $reset_config->getRawData());
  }

}