<?php

namespace Drupal\Tests\disable_account_emails\Unit;

use Drupal\Tests\UnitTestCase;

/**
 * Tests the email types helper function.
 *
 * @group disable_account_emails
 */
class EmailTypesTest extends UnitTestCase {

  /**
   * Tests that the email types function returns the correct structure.
   */
  public function testGetEmailTypes(): void {
    // Include the module file to access the helper function.
    require_once __DIR__ . '/../../../disable_account_emails.module';

    // Call the helper function.
    $email_types = _disable_account_emails_get_email_types();

    // Verify it returns an array.
    $this->assertIsArray($email_types, 'Email types should be an array');

    // Verify it's not empty.
    $this->assertNotEmpty($email_types, 'Email types should not be empty');

    // Verify it has exactly 9 entries.
    $this->assertCount(9, $email_types, 'Should have exactly 9 email types');

    // Verify all expected keys are present.
    $expected_keys = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'register_activate',
      'status_blocked',
      'cancel_confirm',
      'status_deleted',
      'password_reset',
    ];

    foreach ($expected_keys as $key) {
      $this->assertArrayHasKey($key, $email_types, "Key '$key' should exist");
    }

    // Verify all values are translatable strings (not empty).
    foreach ($email_types as $key => $label) {
      $this->assertNotEmpty($label, "Label for '$key' should not be empty");
      $this->assertIsObject($label, "Label for '$key' should be a TranslatableMarkup object");
    }

    // Verify specific expected labels exist.
    $this->assertEquals('Welcome (no approval required)', (string) $email_types['register_no_approval_required']);
    $this->assertEquals('Password recovery', (string) $email_types['password_reset']);
    $this->assertEquals('Account blocked', (string) $email_types['status_blocked']);
  }

  /**
   * Tests that email types keys match Drupal core user module email keys.
   */
  public function testEmailTypesMatchCoreKeys(): void {
    require_once __DIR__ . '/../../../disable_account_emails.module';

    $email_types = _disable_account_emails_get_email_types();

    // These are the actual email keys used by Drupal core's user module.
    // Verify our keys match exactly.
    $core_email_keys = [
      'register_admin_created',
      'register_pending_approval',
      'register_pending_approval_admin',
      'register_no_approval_required',
      'status_activated',
      // Note: 'register_activate' is an alias/variant.
      'status_blocked',
      'cancel_confirm',
      'status_canceled',
      // Note: 'status_deleted' is an alias/variant.
      'password_reset',
    ];

    // Verify our module includes the main email types.
    $this->assertArrayHasKey('register_admin_created', $email_types);
    $this->assertArrayHasKey('register_no_approval_required', $email_types);
    $this->assertArrayHasKey('password_reset', $email_types);
  }

  /**
   * Tests that the function returns consistent results.
   */
  public function testGetEmailTypesConsistency(): void {
    require_once __DIR__ . '/../../../disable_account_emails.module';

    // Call the function multiple times.
    $result1 = _disable_account_emails_get_email_types();
    $result2 = _disable_account_emails_get_email_types();

    // Verify both calls return the same keys.
    $this->assertEquals(array_keys($result1), array_keys($result2), 'Function should return consistent keys');

    // Verify both calls return the same number of items.
    $this->assertCount(count($result1), $result2, 'Function should return consistent count');
  }

}