<?php

namespace Drupal\disable_login\Access;

use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Routing\RouteMatch;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class DisableLoginAccessCheck.
 *
 * Checks access for displaying configuration translation page.
 */
class DisableLoginAccessCheck implements AccessInterface {


  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The config factory.
   *
   * @var Symfony\Component\HttpFoundation\RequestStack
   */
  protected $request;

  /**
   * Constructs an DisableLoginAccessCheck object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param Symfony\Component\HttpFoundation\RequestStack $request
   *   The request stack.
   */
  public function __construct(ConfigFactoryInterface $config_factory, RequestStack $request) {
    $this->configFactory = $config_factory;
    $this->request = $request->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('request_stack')
     );
  }

  /**
   * A custom access check.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   * @param \Drupal\Core\Routing\RouteMatch $route_match
   *   The route which is attempted to be accessed.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, RouteMatch $route_match) {
    $route_name = $route_match->getRouteName();
    switch ($route_name) {
      // For login pages check for token.
      case 'user.login':
      case 'user.login.http':
        if (!$this->hasValidSecretToken($route_match)) {
          return AccessResult::forbidden();
        }
    }
    return AccessResult::allowed();
  }

  /**
   * Check if the URL has a valid secret token.
   *
   * @param \Drupal\Core\Routing\RouteMatch $route_match
   *   The route which is attempted to be accessed.
   *
   * @return bool
   *   TRUE if the URL has valid secret token.
   */
  private function hasValidSecretToken(RouteMatch $route_match) {
    $route_name = $route_match->getRouteName();
    switch ($route_name) {
      // For login pages check for token.
      case 'user.login':
      case 'user.login.http':
        $config = $this->configFactory->get('disable_login.settings');
        // If login pages are protected based on the configuration for
        // the environment, check for key.
        if ($config->get('disable_login')) {
          $key_name = $config->get('querystring');
          $secret_key = $config->get('secret');
          $key_value = $this->request->get($key_name);
          if ($key_value == $secret_key) {
            return TRUE;
          }
          else {
            return FALSE;
          }
        }
      default:
        // Do nothing.
    }
    // Protect only those pages that require the key.
    // Return TRUE by default.
    return TRUE;
  }

}
