<?php

declare(strict_types=1);

namespace Drupal\discourse_comments_plus\Controller;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Returns responses for Discourse comments plus sso routes.
 */
final class DiscourseCommentsSSOController extends ControllerBase {
  /**
   * SSO secret.
   *
   * @var string
   */
  private $ssoSecret;

  /**
   * Base url for discourse.
   *
   * @var string
   */
  private $baseUrl;

  /**
   * Public url for discourse.
   *
   * @var string
   */
  private $publicUrl;

  /**
   * The controller constructor.
   */
  public function __construct(
    ConfigFactory $configFactory,
  ) {
    $discourse_settings = $configFactory->get('discourse_comments_plus.discourse_comments_settings');
    $this->baseUrl = $discourse_settings->get('internal_base_url_of_discourse') ?: $discourse_settings->get('base_url_of_discourse');
    $this->publicUrl = $discourse_settings->get('base_url_of_discourse');
    $this->ssoSecret = $discourse_settings->get('sso_secret');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
    );
  }

  /**
   * Builds the response.
   */
  public function __invoke() {
    $request = \Drupal::request();
    $parameters = $request->query->all();

    if (isset($parameters['sso']) && isset($parameters['sig'])) {
      $sso = $parameters['sso'];
      $incoming_signature = $parameters['sig'];
      $calculated_signature = hash_hmac('sha256', $sso, $this->ssoSecret);
      if (hash_equals($calculated_signature, $incoming_signature)) {
        $sso = urldecode($sso);
        $query = [];
        parse_str(base64_decode($sso), $query);

        $session = $request->getSession();
        $session->set('discourse_comments_plus_sso', $query);
        $sso_url_query_parts = parse_url($query['return_sso_url']);
        $sso_url_query = [];
        parse_str($sso_url_query_parts['query'], $sso_url_query);

        return new TrustedRedirectResponse($sso_url_query['d'] ?? Url::fromRoute('<front>')->toString());
      }
      else {
        throw new NotFoundHttpException();
      }
    }
    else {
      $nonce = hash('sha512', (string) mt_rand());
      $return_url = Url::fromRoute('discourse_comments_plus.discourse_comments_plus_sso', [], [
        'absolute' => TRUE,
        'query' => [
          'd' => $parameters['d'] ?? '<front>',
        ],
      ])->toString();

      $payload = base64_encode(http_build_query(
        [
          'nonce' => $nonce,
          'return_sso_url' => $return_url
        ]
      ));

      $request = array(
        'sso' => $payload,
        'sig' => hash_hmac('sha256', $payload, $this->ssoSecret)
      );

      $query = http_build_query($request);
      $url = $this->publicUrl . '/session/sso_provider?' . $query;

      return new TrustedRedirectResponse($url);
    }

  }

}
