<?php

namespace Drupal\discourse_comments_plus;

use Drupal\Core\Utility\Error;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannelTrait;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\RequestException;

/**
 * Class DiscourseApiClient.
 *
 * @package Drupal\discourse_comments_plus
 */
class DiscourseApiClient {

  use LoggerChannelTrait;

  /**
   * HTTP client factory.
   *
   * @var \GuzzleHttp\Client
   */
  protected $client;

  /**
   * Api headers.
   *
   * @var array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|null
   */
  private $apiHeaders;

  /**
   * Base url for discourse.
   *
   * @var array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|null
   */
  private $baseUrl;

  /**
   * Public url for discourse.
   *
   * @var array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|null
   */
  private $publicUrl;

  /**
   * Cache backend service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * EntityTypeManager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Database service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The extension path resolver.
   *
   * @var \Drupal\Core\Extension\ExtensionPathResolver
   */
  protected $extensionPathResolver;

  /**
   * Original API headers for restoration.
   *
   * @var array
   */
  protected $originalApiHeaders;

  /**
   * Original base URL for restoration.
   *
   * @var string
   */
  protected $originalBaseUrl;

  /**
   * Cache suffix for domain configuration.
   *
   * @var string
   */
  protected $cacheSuffix = 'global_';

  /**
   * CatFactsClient constructor.
   *
   * @param \Drupal\Core\Http\ClientFactory $http_client_factory
   *   Http client factory.
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   Config factory service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Time service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   EntityTypeManager service.
   * @param \Drupal\Core\Database\Connection $database
   *   Database service.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extension_path_resolver
   *   The extension path resolver service.
   */
  public function __construct(
    ClientFactory $http_client_factory,
    ConfigFactory $config_factory,
    CacheBackendInterface $cacheBackend,
    TimeInterface $time,
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    ExtensionPathResolver $extension_path_resolver,
  ) {
    $discourseSettings = $config_factory->get('discourse_comments_plus.discourse_comments_settings');
    try {
      $this->baseUrl = $discourseSettings->get('internal_base_url_of_discourse') ?: $discourseSettings->get('base_url_of_discourse');
      $this->publicUrl = $discourseSettings->get('base_url_of_discourse');
      $this->apiHeaders = [
        'Api-Key' => $discourseSettings->get('api_key'),
        'Api-Username' => $discourseSettings->get('api_user_name'),
      ];

      $this->client = $http_client_factory->fromOptions([
        'base_uri' => $this->baseUrl,
        'timeout' => 30,
      ]);
      $this->cache = $cacheBackend;
      $this->time = $time;
      $this->configFactory = $discourseSettings;
      $this->entityTypeManager = $entity_type_manager;
      $this->database = $database;
      $this->logger = $this->getLogger('discourse_comments_plus');
      $this->extensionPathResolver = $extension_path_resolver;

      // Store original values for restoration after temporary use.
      $this->originalBaseUrl = $this->baseUrl;
      $this->originalApiHeaders = $this->apiHeaders;

      if (\Drupal::moduleHandler()->moduleExists('domain_config')) {
        /** @var \Drupal\domain_config\Config\DomainConfigFactoryOverrideInterface */
        $domain_config_overrider = \Drupal::getContainer()->get('domain.config_factory_override');
        $this->cacheSuffix = $domain_config_overrider->getCacheSuffix() . '_';
      }
    }
    catch (ConnectException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
  }

  /**
   * Get topic by topic id.
   *
   * @param int $topic_id
   *   Topic id.
   *
   * @return string|bool
   *   Returns topic data.
   */
  public function getTopic(int $topic_id) {
    $uri = sprintf('/t/%s.json', $topic_id);
    try {
      $response = $this->client->get($uri, [
        'base_uri' => $this->baseUrl,
        'headers' => $this->apiHeaders,
      ]);

      return $response->getBody()->getContents();
    }
    catch (ConnectException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (ClientException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (RequestException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }

    return FALSE;
  }

  /**
   * Get list of categories from discourse.
   *
   * @param bool $ignore_cache
   *   If TRUE, ignore cache and fetch fresh data.
   *
   * @return \Psr\Http\Message\StreamInterface|bool
   *   Returns list of categories from discourse.
   */
  public function getCategories($ignore_cache = FALSE) {
    if (($categories = $this->cache->get($this->cacheSuffix . 'discourse_category')) && !$ignore_cache) {
      return $categories->data;
    }
    $uri = sprintf('/categories.json');
    try {
      $response = $this->client->get($uri, [
        'base_uri' => $this->baseUrl,
        'headers' => $this->apiHeaders,
      ]);

      $data = Json::decode($response->getBody());
      $time_value = $this->time->getCurrentTime();
      // 12 hours cache time for categories data.
      $this->cache->set($this->cacheSuffix . 'discourse_category', $data, $time_value + 43200);
      return $data;
    }
    catch (ConnectException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    return FALSE;
  }

  /**
   * Post topic to discourse.
   *
   * @param array $data
   *   Post data.
   * @param string $user
   *   Username.
   *
   * @return string|\Exception
   *   Returns newly created post data.
   */
  public function post(array $data, $user = NULL) {
    $headers = $this->apiHeaders;

    if ($user !== NULL) {
      $headers['Api-Username'] = $user;
    }

    $headers['Content-Type'] = 'multipart/form-data';
    $headers['Accept'] = 'application/json; charset=utf-8';
    $headers['content-encoding'] = 'gzip';
    $uri = '/posts.json';
    try {
      $response = $this->client->post($uri, [
        'base_uri' => $this->baseUrl,
        'form_params' => $data,
        'headers' => $headers,
      ]);
      return $response->getBody()->getContents();
    }
    catch (ConnectException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
      return $e;
    }
    catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
      return $e;
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
      return $e;
    }
  }

  /**
   * Returns base url.
   *
   * @return array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|mixed|null
   *   Base url for discourse.
   */
  public function getBaseUrl() {
    return $this->baseUrl;
  }

  /**
   * Returns public url.
   *
   * @return array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|mixed|null
   *   Base url for discourse.
   */
  public function getPublicUrl() {
    return $this->publicUrl;
  }

  /**
   * Get list of latest comments from discourse.
   *
   * @param int $count
   *   Number of comments to display.
   * @param int $before
   *   Get older posts before post id.
   * @param array $latest_comments_data
   *   Latest comments array. Used in subsequent request if less then 5
   *   comments present for corresponding node.
   * @param int $pass
   *   Number of time discourse api is called for getting latest posts.
   *
   * @return array
   *   Returns latest comments from discourse.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getLatestComments($count = 5, $before = 0, array $latest_comments_data = [], $pass = 1) {
    if ($latest_comments = $this->cache->get($this->cacheSuffix . 'discourse_latest_comments')) {
      return $latest_comments->data;
    }
    else {
      return FALSE;
    }
  }

  /**
   * Get node id from topic id.
   *
   * @param int $topic_id
   *   Topic id from discourse.
   *
   * @return \Drupal\Core\Entity\EntityInterface|bool
   *   Nid corresponding to topic id.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getNodeFromTopicId($topic_id) {
    $query = $this->database->select('node_field_data', 'nf');
    $query->addField('nf', 'nid');
    $query->condition('nf.discourse_plus_field__topic_id', $topic_id);
    $results = $query->execute();
    $results = $results->fetch();
    $nid = 0;
    if (isset($results->nid)) {
      $nid = $results->nid;
    }

    if ($nid) {
      $node_storage = $this->entityTypeManager->getStorage('node');
      $node = $node_storage->load($nid);
      return $node;
    }
    else {
      return FALSE;
    }
  }

  /**
   * Get default avatar image path.
   *
   * @return string
   *   Returns default avatar image path.
   */
  public function getDefaultAvatar() {
    return sprintf('/%s/%s', $this->extensionPathResolver->getPath('module', 'discourse_comments_plus'), 'images/user-default.png');
  }

  /**
   * Get header for discourse api client.
   *
   * @return array|\Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig|null
   *   Returns api headers for discourse client.
   */
  public function getHeaders() {
    return $this->apiHeaders;
  }

  /**
   * Get the client for sending api requests.
   *
   * @return \GuzzleHttp\Client
   *   Returns the client for api requests.
   */
  public function getClient() {
    return $this->client;
  }

  /**
   * Get topic ids which has disscourse comment count.
   */
  public function getTopicIdsWithComments() {
    $query = $this->database->select('node_field_data', 'nf');
    $query->addField('nf', 'discourse_plus_field__topic_id');
    $query->condition('nf.discourse_plus_field__topic_id', NULL, 'IS NOT NULL');
    $query->condition('nf.status', 1);
    $query->orderBy('created', 'DESC');
    $query->range(0, 20);
    $results = $query->execute();
    $results = $results->fetchAll();
    $topic_ids = [];
    foreach ($results as $record) {
      $topic_ids[] = $record->discourse_plus_field__topic_id;
    }

    if (count($topic_ids) > 0) {
      return $topic_ids;
    }
    else {
      return FALSE;
    }
  }

  /**
   * Fetch latest comments from discourse and set cache.
   *
   * @param int $count
   *   Number of comments to fetch.
   *
   * @return array|bool
   *   Returns latest comments data.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function fetchLatestComments($count = 5) {
    try {
      // Get top 20 topic ids for checking latest comments.
      $topic_ids = $this->getTopicIdsWithComments();

      $all_comments = [];
      // Generate array for all comments.
      foreach ($topic_ids as $topic) {
        if (is_numeric($topic)) {
          $d_topic = Json::decode($this->getTopic($topic));
          if (isset($d_topic['post_stream']) && isset($d_topic['post_stream']['posts'])) {
            foreach ($d_topic['post_stream']['posts'] as $key => $comment) {
              // Forst entry is topic so skip it, we just want comments.
              if ($key > 0) {
                // Skip if user have deleted the comment.
                if ($comment['user_deleted']) {
                  continue;
                }
                $comment_time = strtotime($comment['created_at']);

                $all_comments[$comment_time]['id'] = $comment['id'];
                $all_comments[$comment_time]['username'] = $comment['username'];
                $all_comments[$comment_time]['topic_id'] = $comment['topic_id'];
                $all_comments[$comment_time]['user_deleted'] = $comment['user_deleted'];
                $all_comments[$comment_time]['avatar_template'] = $comment['avatar_template'];
                $all_comments[$comment_time]['post_content'] = Unicode::truncate(strip_tags($comment['cooked']), 100, TRUE, TRUE, 10);
                $all_comments[$comment_time]['created_at'] = strtotime($comment['created_at']);
              }
            }
          }
        }
      }
      // Sort comments so we can pick latest 5.
      krsort($all_comments, SORT_NUMERIC);
      // Loop and pick top 5 comments.
      foreach ($all_comments as $comment) {
        // Remove deleted comments.
        if ($comment['user_deleted']) {
          continue;
        }

        $default_avatar_image = $this->getDefaultAvatar();
        // Appending base url if https:// does not exist in image path.
        if (strpos($comment['avatar_template'], "https://") === FALSE) {
          $avatar_image = sprintf('%s%s', $this->getBaseUrl(), str_replace('{size}', '90', $comment['avatar_template']));
        }
        else {
          $avatar_image = str_replace('{size}', '90', $comment['avatar_template']);
        }
        // Placing default avatar image if avatar image does not exist.
        if (@getimagesize($avatar_image)) {
          $latest_comments[$comment['id']]['avatar_template'] = sprintf('%s%s', $this->getPublicUrl(), str_replace('{size}', '90', $comment['avatar_template']));
        }
        else {
          $latest_comments[$comment['id']]['avatar_template'] = $default_avatar_image;
        }

        $latest_comments[$comment['id']]['username'] = $comment['username'];

        // Set comment url.
        $comment_node = $this->getNodeFromTopicId($comment['topic_id']);
        $link = $comment_node->toUrl()->toString();
        $latest_comments[$comment['id']]['comment_url'] = sprintf("%s#discourse-comment", $link);

        $latest_comments[$comment['id']]['post_content'] = $comment['post_content'];

        if (count($latest_comments) >= $count) {
          break;
        }
      }
      $time_value = $this->time->getCurrentTime();
      // Convert cache_lifetime in seconds.
      $cache_lifetime = $this->configFactory->get('cache_lifetime') * 60;
      // Set cache time for discourse latest comments.
      $this->cache->set($this->cacheSuffix . 'discourse_latest_comments', $latest_comments, $time_value + $cache_lifetime);
      return $latest_comments;
    }
    catch (ConnectException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage(), Error::decodeException($e));
      return FALSE;
    }
    return FALSE;
  }

  /**
   * Temporarily sets the API credentials for a single request sequence.
   *
   * This is useful for testing unsaved credentials from a form.
   *
   * @param string $base_url
   *   The Discourse base URL.
   * @param string $api_key
   *   The Discourse API key.
   * @param string $api_user
   *   The Discourse API username.
   */
  public function setTemporaryCredentials(string $base_url, string $api_key, string $api_user) {
    $this->baseUrl = $base_url;
    $this->apiHeaders['Api-Key'] = $api_key;
    $this->apiHeaders['Api-Username'] = $api_user;
  }

  /**
   * Restores the original API credentials after a temporary override.
   */
  public function restoreOriginalCredentials() {
    $this->baseUrl = $this->originalBaseUrl;
    $this->apiHeaders = $this->originalApiHeaders;
  }
}
