<?php

namespace Drupal\discourse_comments_plus\Plugin\Block;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\node\NodeInterface;

/**
 * Provides a 'DiscourseCommentPlusBlock' block.
 *
 * @Block(
 *  id = "discourse_comment_plus_block",
 *  admin_label = @Translation("Discourse comment block"),
 * )
 */
class DiscourseCommentBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Drupal\discourse_comments_plus\DiscourseApiClient definition.
   *
   * @var \Drupal\discourse_comments_plus\DiscourseApiClient
   */
  protected $discourseApiClient;

  /**
   * Route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatter
   */
  protected $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->discourseApiClient = $container->get('discourse_comments_plus.discourse_api_client');
    $instance->currentRouteMatch = $container->get('current_route_match');
    $instance->configFactory = $container->get('config.factory');
    $instance->dateFormatter = $container->get('date.formatter');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    /** @var \Drupal\node\NodeInterface $node */
    $node = $this->currentRouteMatch->getParameter('node');
    $discourse_settings = $this->configFactory->get('discourse_comments_plus.discourse_comments_settings');
    $cache_tags = $discourse_settings->getCacheTags();

    if ($node instanceof NodeInterface && $node->hasField('discourse_plus_field')) {
      $cache_tags = Cache::mergeTags($cache_tags, $node->getCacheTags());
      $field_discourse = $node->get('discourse_plus_field')->getValue();
      if (isset($field_discourse[0]['topic_id']) && is_numeric($field_discourse[0]['topic_id'])) {
        $topic_id = $field_discourse[0]['topic_id'];
        $topic = Json::decode($this->discourseApiClient->getTopic($field_discourse[0]['topic_id']));
        $cache_tags = Cache::mergeTags($cache_tags, ['discourse_comments_plus.topic_id.' . $topic_id]);
        // Save topic count.
        $post_count = ($topic['posts_count'] > 0) ? ($topic['posts_count'] - 1) : 0;
        $field_discourse[0]['comment_count'] = $post_count;
        $node->set('discourse_plus_field', $field_discourse)->save();
        $comments = [];
        $default_avatar_image = $this->discourseApiClient->getDefaultAvatar();
        if (isset($topic['post_stream']) && isset($topic['post_stream']['posts'])) {
          foreach ($topic['post_stream']['posts'] as $key => $post) {
            if ($key > 0) {
              if ($post['user_deleted']) {
                continue;
              }
              $comments[$key]['username'] = $post['username'];

              // Appending base url if https:// does not exist in image path.
              if (strpos($post['avatar_template'], "https://") === FALSE) {
                $avatar_image = sprintf('%s%s', $this->discourseApiClient->getBaseUrl(), str_replace('{size}', '90', $post['avatar_template']));
              }
              else {
                $avatar_image = str_replace('{size}', '90', $post['avatar_template']);
              }

              // Placing default avatar image if avatar image does not exist.
              if (@getimagesize($avatar_image)) {
                $comments[$key]['avatar_template'] = sprintf('%s%s', $this->discourseApiClient->getPublicUrl(), str_replace('{size}', '90', $post['avatar_template']));
              }
              else {
                $comments[$key]['avatar_template'] = $default_avatar_image;
              }

              $date = $this->dateFormatter->format(strtotime($post['created_at']), 'custom', 'F d, Y');
              $comments[$key]['created_at'] = $date;
              $comments[$key]['post_content'] = $post['cooked'];
            }
          }
        }

        $build['#theme'] = 'discourse_comment_block';
        $build['#content'] = $comments;
        $build['#topic_url'] = $field_discourse[0]['topic_url'];

        if (\Drupal::request()->getSession()->has('discourse_comments_plus_sso')) {
          $comment_form = \Drupal::formBuilder()->getForm('Drupal\discourse_comments_plus\Form\DiscourseCommentForm', $node);
          $build['#form'] = $comment_form;
        }
        else {
          $build['#sso'] = Link::createFromRoute('Login to comment', 'discourse_comments_plus.discourse_comments_plus_sso', ['d' => \Drupal::request()->getRequestUri(),], )->toRenderable();
        }
      }
    }

    $cache_lifetime = $discourse_settings->get('cache_lifetime');
    $build['#cache'] = [
      'tags' => $cache_tags,
      'contexts' => ['url.path', 'session'],
      'max-age' => $cache_lifetime * 60,
    ];

    return $build;
  }

}
