<?php

declare(strict_types=1);

namespace Drupal\discourse_comments_plus\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Discourse comments form.
 */
final class DiscourseCommentForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'discourse_comments_plus_comment';
  }

  /**
   * Drupal\discourse_comments_plus\DiscourseApiClient definition.
   *
   * @var \Drupal\discourse_comments_plus\DiscourseApiClient
   */
  protected $discourseApiClient;

  /**
   * Current session.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface;
   */
  protected $currentSession;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->discourseApiClient = $container->get('discourse_comments_plus.discourse_api_client');
    $instance->currentSession = $container->get('request_stack')->getCurrentRequest()->getSession();

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#required' => TRUE,
      '#attributes' => [
        'minlength' => 10,
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Comment'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (!$this->currentSession->has('discourse_comments_plus_sso')) {
      $form_state->setErrorByName(
        'message',
        $this->t('Unexpected error occurred. Please try again.'),
      );
      $form_state->setRebuild();
    }
    if (mb_strlen($form_state->getValue('message')) < 10) {
      $form_state->setErrorByName(
        'message',
        $this->t('Message should be at least 10 characters.'),
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\node\Entity\Node $node */
    $node = $form_state->getBuildInfo()['args'][0];
    $discourse_topic_id = $node->get('discourse_plus_field')->topic_id;
    $message = $form_state->getValue('message');
    $footer_markup = _discourse_comments_plus_footer_markup($node);
    $body = $message . $footer_markup;
    $sso = $this->currentSession->get('discourse_comments_plus_sso');

    $response = $this->discourseApiClient->post([
      'topic_id' => $discourse_topic_id,
      'raw' => $body,
    ], $sso['username']);

    if ($response instanceof \Exception) {
      $this->messenger()->addError($this->t('An error occurred while sending the comment.'));
      return;
    }

    Cache::invalidateTags(['discourse_comments_plus.topic_id.' . $discourse_topic_id]);
    $this->messenger()->addStatus($this->t('The comment has been sent.'));
  }

}
