<?php

namespace Drupal\discourse_comments_plus\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Class DiscourseCommentsSettingsForm.
 */
class DiscourseCommentsSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The discourse API client.
   *
   * @var \Drupal\discourse_comments_plus\DiscourseApiClient
   */
  protected $discourseApiClient;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->discourseApiClient = $container->get('discourse_comments_plus.discourse_api_client');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->token = $container->get('token');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'discourse_comments_plus.discourse_comments_settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'discourse_comments_plus_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('discourse_comments_plus.discourse_comments_settings');
    $form['base_url_of_discourse'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Discourse Base URL'),
      '#description' => $this->t('Please enter url without trailing / character. Example: https://test.trydiscourse.com'),
      '#maxlength' => 256,
      '#size' => 64,
      '#default_value' => $config->get('base_url_of_discourse'),
      '#required' => TRUE,
    ];

    $form['checkbox_internal_base_url_of_discourse'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Internal Discourse Base URL is different'),
      '#default_value' => $config->get('checkbox_internal_base_url_of_discourse'),
    ];

    $form['internal_base_url_of_discourse'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Discourse Internal Base URL'),
      '#description' => $this->t('Please enter url without trailing / character. Example: http://discourse:3000'),
      '#maxlength' => 256,
      '#size' => 64,
      '#default_value' => $config->get('internal_base_url_of_discourse'),
      '#states' => [
        'visible' => [
          ':input[name="checkbox_internal_base_url_of_discourse"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="checkbox_internal_base_url_of_discourse"]' => ['checked' => TRUE],
        ],
      ]
    ];

    $form['sso_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Discourse SSO Secret'),
      '#description' => $this->t('Make sure to use the same secret in Discourse and Drupal. To configure SSO, go to <br><code>/admin/site_settings/category/login?filter=discourse%20connect%20provider</code>'),
      '#maxlength' => 256,
      '#size' => 64,
      '#required' => TRUE,
      '#default_value' => $config->get('sso_secret'),
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Discourse Api Key'),
      '#maxlength' => 256,
      '#size' => 64,
      '#required' => TRUE,
      '#default_value' => $config->get('api_key'),
    ];

    $form['api_user_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Discourse Api Username'),
      '#maxlength' => 256,
      '#size' => 64,
      '#required' => TRUE,
      '#default_value' => $config->get('api_user_name'),
    ];

    $form['cache_lifetime'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Lifetime (in minutes)'),
      '#size' => 64,
      '#required' => TRUE,
      // Default cache lifetime 60 minutes.
      '#default_value' => $config->get('cache_lifetime') ? $config->get('cache_lifetime') : 60,
    ];

    $form['footer_template'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Footer Template'),
      '#format' => 'full_html',
      '#rows' => 2,
      '#required' => FALSE,
      '#default_value' => $config->get('footer_template'),
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author], etc.'),
    ];

    if ($this->moduleHandler->moduleExists('token')) {
      $form['footer_template']['#description'] .= ' ' . $this->t('Click the "Browse available tokens" button to see all available tokens.');
      $form['footer_template']['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['node'],
        '#show_restricted' => FALSE,
        '#global_types' => FALSE,
        '#weight' => 90,
      ];
    }

    $form['category_settings_wrapper'] = [
      '#type' => 'details',
      '#title' => $this->t('Category Settings'),
      '#open' => TRUE,
      '#prefix' => '<div id="discourse-category-settings-wrapper">',
      '#suffix' => '</div>',
    ];

    $all_content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $content_type_options = array_map(function ($type) {
      return $type->label();
    }, $all_content_types);

    $base_url = $config->get('base_url_of_discourse');
    if ($config->get('checkbox_internal_base_url_of_discourse')) {
      $base_url = $config->get('internal_base_url_of_discourse');
    }
    $api_key = $config->get('api_key');
    $api_user = $config->get('api_user_name');

    if (!empty($base_url) && !empty($api_key) && !empty($api_user)) {
      $categories = $this->discourseApiClient->getCategories();
      $category_options = [];

      if ($categories && !empty($categories['category_list']['categories'])) {
        foreach ($categories['category_list']['categories'] as $cat) {
          $category_options[$cat['id']] = $cat['name'];
        }

        $triggering_element = $form_state->getTriggeringElement();

        if ($triggering_element && isset($triggering_element['#ajax']['callback']) && $triggering_element['#ajax']['callback'][1] === 'refreshCategoriesCallback') {
          \Drupal::messenger()->addStatus($this->t('Successfully refreshed categories from Discourse.'));
        }
      }
      else {
        $form['category_settings_wrapper']['category_error'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--warning">' . $this->t('Could not fetch categories from Discourse. Please enter and save your API credentials, then click "Refresh Category List".') . '</div>',
        ];
      }

      $form['category_settings_wrapper']['refresh_categories'] = [
        '#type' => 'button',
        '#value' => $this->t('Refresh Category List'),
        '#ajax' => [
          'callback' => '::refreshCategoriesCallback',
          'wrapper' => 'discourse-category-settings-wrapper',
          'progress' => ['type' => 'throbber', 'message' => $this->t('Fetching categories...')],
        ],
        '#weight' => -10,
      ];

      if (!empty($category_options)) {
        $form['category_settings_wrapper']['default_category'] = [
          '#type' => 'select',
          '#title' => $this->t('Default category to post to'),
          '#options' => $category_options,
          '#default_value' => $config->get('default_category'),
          '#empty_option' => $this->t('- Select a category -'),
        ];

        // AJAX wrapper for the overrides section.
        $form['category_settings_wrapper']['overrides_container'] = [
          '#type' => 'details',
          '#title' => $this->t('Category Overrides'),
          '#open' => TRUE,
          '#description' => $this->t('Add and configure category overrides for specific content types.'),
          '#prefix' => '<div id="discourse-overrides-wrapper">',
          '#suffix' => '</div>',
        ];

        if ($form_state->get('configured_overrides') === NULL) {
          $configured_overrides = array_keys(array_filter($config->get('overridden_default_category_content_types') ?: []));
          $form_state->set('configured_overrides', $configured_overrides);
        }
        $configured_overrides = $form_state->get('configured_overrides');

        $form['category_settings_wrapper']['overrides_container']['overridden_default_category_options'] = [
          '#type' => 'container',
          '#tree' => TRUE,
        ];

        if (!empty($configured_overrides)) {
          foreach ($configured_overrides as $content_type_id) {
            if (isset($content_type_options[$content_type_id])) {
              $form['category_settings_wrapper']['overrides_container']['overridden_default_category_options'][$content_type_id] = [
                '#type' => 'select',
                '#title' => $this->t('Override for @content_type', ['@content_type' => $content_type_options[$content_type_id]]),
                '#options' => $category_options,
                '#default_value' => $config->get("overridden_default_category_options.{$content_type_id}"),
                '#empty_option' => $this->t('- No override -'),
              ];
            }
          }
        }
        else {
          $form['category_settings_wrapper']['overrides_container']['no_overrides'] = [
            '#markup' => '<p>' . $this->t('No overrides have been added yet.') . '</p>',
          ];
        }

        $available_options = array_diff_key($content_type_options, array_flip($configured_overrides));
        if (!empty($available_options)) {
          $form['category_settings_wrapper']['overrides_container']['add_override'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['container-inline']],
            'new_override_content_type' => [
              '#type' => 'select',
              '#title' => $this->t('Add override for'),
              '#options' => $available_options,
              '#empty_option' => $this->t('- Select content type -'),
            ],
            'add_button' => [
              '#type' => 'submit',
              '#value' => $this->t('Add'),
              '#submit' => ['::addOverrideSubmitHandler'],
              '#ajax' => ['callback' => '::overridesAjaxCallback', 'wrapper' => 'discourse-overrides-wrapper'],
            ],
          ];
        }
      }
    }
    else {
      // Credentials are not configured yet.
      $form['category_settings']['category_error'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' . $this->t('Please enter and save your Discourse API credentials to load the category list.') . '</div>',
      ];
    }

    $form['content_types_enabled_for_discourse'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types for which Discourse Comments Integration should be enabled by default'),
      '#options' => $content_type_options,
      '#default_value' => $config->get('content_types_enabled_for_discourse') == NULL ? [] : $config->get('content_types_enabled_for_discourse'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to refresh the category list from Discourse.
   */
  public function refreshCategoriesCallback(array &$form, FormStateInterface $form_state) {
    $base_url = $form_state->getValue('base_url_of_discourse');
    if ($form_state->getValue('checkbox_internal_base_url_of_discourse')) {
      $base_url = $form_state->getValue('internal_base_url_of_discourse');
    }
    $api_key = $form_state->getValue('api_key');
    $api_user = $form_state->getValue('api_user_name');

    $category_settings_element = &$form['category_settings_wrapper'];
    unset($category_settings_element['default_category'], $category_settings_element['category_error']);

    if (empty($base_url) || empty($api_key) || empty($api_user)) {
      $category_settings_element['category_error'] = [
        '#markup' => '<div class="messages messages--error">' . $this->t('Please fill in the Base URL, API Key, and API Username fields to test the connection.') . '</div>',
      ];
      return $category_settings_element;
    }

    try {
      $this->discourseApiClient->setTemporaryCredentials($base_url, $api_key, $api_user);
      $categories = $this->discourseApiClient->getCategories(TRUE);

      if ($categories && !empty($categories['category_list']['categories'])) {
        $options = [];
        foreach ($categories['category_list']['categories'] as $cat) {
          $options[$cat['id']] = $cat['name'];
        }
        $category_settings_element['default_category'] = [
          '#type' => 'select',
          '#title' => $this->t('Default category to post to'),
          '#options' => $options,
          '#default_value' => $form_state->getValue('default_category'),
          '#empty_option' => $this->t('- Select a category -'),
        ];
        \Drupal::messenger()->addStatus($this->t('Successfully refreshed categories from Discourse.'));
      }
      else {
        $category_settings_element['category_error'] = [
          '#markup' => '<div class="messages messages--error">' . $this->t('Connection failed or no categories found. Please verify your API settings are correct.') . '</div>',
        ];
      }
    }
    finally {
      $this->discourseApiClient->restoreOriginalCredentials();
    }

    return $category_settings_element;
  }


  /**
   * AJAX callback to rebuild the overrides section.
   */
  public function overridesAjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['category_settings_wrapper']['overrides_container'];
  }

  /**
   * Submit handler for adding a new override.
   */
  public function addOverrideSubmitHandler(array &$form, FormStateInterface $form_state) {
    $new_override = $form_state->getValue('new_override_content_type');
    if (!empty($new_override)) {
      $configured = $form_state->get('configured_overrides');
      $configured[] = $new_override;
      $form_state->set('configured_overrides', array_unique($configured));
    }
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $overridden_options = $form_state->getValue('overridden_default_category_options');
    $active_overrides = is_array($overridden_options) ? array_filter($overridden_options) : [];
    $overridden_content_types = array_combine(array_keys($active_overrides), array_keys($active_overrides));

    $this->config('discourse_comments_plus.discourse_comments_settings')
      ->set('base_url_of_discourse', $form_state->getValue('base_url_of_discourse'))
      ->set('checkbox_internal_base_url_of_discourse', $form_state->getValue('checkbox_internal_base_url_of_discourse'))
      ->set('internal_base_url_of_discourse', $form_state->getValue('internal_base_url_of_discourse'))
      ->set('sso_secret', $form_state->getValue('sso_secret'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('api_user_name', $form_state->getValue('api_user_name'))
      ->set('cache_lifetime', $form_state->getValue('cache_lifetime'))
      ->set('footer_template', $form_state->getValue('footer_template')['value'])
      ->set('default_category', $form_state->getValue('default_category'))
      ->set('overridden_default_category_content_types', $overridden_content_types)
      ->set('overridden_default_category_options', $overridden_options)
      ->set('content_types_enabled_for_discourse', array_filter($form_state->getValue('content_types_enabled_for_discourse')))
      ->save();

    // Reset the form state so it reloads from config next time.
    $form_state->set('configured_overrides', NULL);
  }

}
