<?php

declare(strict_types=1);

namespace Drupal\display_builder;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines the access control handler for the instance entity type.
 *
 * @see https://www.drupal.org/project/coder/issues/3185082
 */
final class InstanceAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResult {
    if ($account->hasPermission((string) $this->entityType->getAdminPermission())) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    return match ($operation) {
      'view' => AccessResult::allowedIfHasPermission($account, 'view display_builder_instance'),
      'update' => AccessResult::allowedIfHasPermission($account, 'edit display_builder_instance'),
      'delete' => AccessResult::allowedIfHasPermission($account, 'delete display_builder_instance'),
      default => AccessResult::neutral(),
    };
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResult {
    $permissions = ['create display_builder_instance', 'administer display_builder_instance'];

    return AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');
  }

}
