<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Returns responses for Display Builder ui routes.
 */
class DisplayBuilderDevToolsController extends ControllerBase {

  /**
   * Load a display builder instance by id.
   *
   * @param string $instance_id
   *   The display builder instance id.
   *
   * @return array
   *   The display build.
   */
  public function view(string $instance_id): array {
    $storage = $this->entityTypeManager()->getStorage('display_builder_instance');

    /** @var \Drupal\display_builder\InstanceInterface $instance */
    $instance = $storage->load($instance_id);

    if (!$instance) {
      return [];
    }
    $profile_id = (string) $instance->getProfile()->id();

    $storage = $this->entityTypeManager()->getStorage('display_builder_profile');
    /** @var \Drupal\display_builder\ProfileInterface $profile */
    $profile = $storage->load($profile_id);

    if ($profile) {
      $view_builder = $this->entityTypeManager()->getViewBuilder('display_builder_profile');

      return $view_builder->view($profile, $instance_id);
    }

    return [
      '#plain_text' => $this->t('Missing @instance_id config.', ['@instance_id' => $instance_id]),
    ];
  }

  /**
   * Delete a sample entity generated by UI Patterns.
   *
   * @param string $entity_type_id
   *   The entity type id.
   * @param string $bundle
   *   The bundle id.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A message and redirect.
   */
  public function deleteSample(string $entity_type_id, string $bundle): RedirectResponse {
    /** @var \Drupal\ui_patterns\Entity\SampleEntityGenerator $sample */
    $sample = \Drupal::service('ui_patterns.sample_entity_generator'); // phpcs:ignore
    $sample->delete($entity_type_id, $bundle);

    $params = ['@entity_type_id' => $entity_type_id, '@bundle' => $bundle];
    $this->messenger()->addStatus($this->t('Sample @entity_type_id @bundle deleted!', $params));

    $redirect = $this->redirect('entity.display_builder_instance.collection');
    $redirect->send();

    return $redirect;
  }

}
