<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Form;

use Drupal\Component\Serialization\Json;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\display_builder\ConfigFormBuilderInterface;
use Drupal\display_builder_dev_tools\MockEntity;
use Drupal\display_builder_entity_view\Entity\EntityViewDisplay;
use Drupal\display_builder_entity_view\Field\DisplayBuilderItemList;
use Drupal\display_builder_page_layout\Entity\PageLayout;
use Drupal\display_builder_views\Plugin\views\display_extender\DisplayExtender;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * Defines an edit display builder instance form.
 */
final class EditForm extends FormBase {

  use AutowireTrait;

  public function __construct(
    protected ConfigFormBuilderInterface $configFormBuilder,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $instance_id = NULL, ?string $routeName = NULL, ?ParameterBag $routeParameters = NULL): array {
    // @todo have a way for saving.
    if (PageLayout::checkInstanceId($instance_id) || EntityViewDisplay::checkInstanceId($instance_id) || DisplayExtender::checkInstanceId($instance_id) || DisplayBuilderItemList::checkInstanceId($instance_id)) {
      return [
        '#markup' => $this->t('This configuration must be managed directly from the entity type.'),
      ];
    }

    $storage = $this->entityTypeManager->getStorage('display_builder_instance');

    if ($storage->load($instance_id) === NULL) {
      $this->messenger()->addError($this->t('Invalid instance id: @id', ['@id' => $instance_id]));
      $form['cancel'] = [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => new Url('entity.display_builder_instance.collection'),
      ];

      return $form;
    }

    if (!$routeName) {
      $routeName = 'display_builder_dev_tools.view';
      $routeParameters = ['instance_id' => $instance_id];
    }

    $display_builder = new MockEntity($instance_id);
    $form[ConfigFormBuilderInterface::PROFILE_PROPERTY] = $this->configFormBuilder->build($display_builder);
    unset($form[ConfigFormBuilderInterface::PROFILE_PROPERTY]['link'], $form[ConfigFormBuilderInterface::PROFILE_PROPERTY]['admin_link']);

    $form['instance_id'] = [
      '#type' => 'hidden',
      '#value' => $instance_id,
    ];

    $form['route_name'] = [
      '#type' => 'hidden',
      '#value' => $routeName,
    ];

    if ($routeParameters) {
      $parameters = [];

      foreach ($routeParameters as $key => $parameter) {
        $parameters[$key] = $parameter;
      }
      $form['route_parameters'] = [
        '#type' => 'hidden',
        '#value' => Json::encode($parameters),
      ];
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => new Url('entity.display_builder_instance.collection'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'display_builder_dev_tools_reset';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entity_config_id = $form_state->getValue(ConfigFormBuilderInterface::PROFILE_PROPERTY);
    $instance_id = $form_state->getValue('instance_id');
    $storage = $this->entityTypeManager->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->load($instance_id);

    $instance->setProfile($entity_config_id);
    $instance->save();
    // phpcs:ignore
    \Drupal::service('plugin.cache_clearer')->clearCachedDefinitions();

    $display_builder = new MockEntity($instance_id);
    $display_builder->initInstanceIfMissing();
    $form_state->setRedirectUrl($display_builder->getBuilderUrl());
  }

}
