<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Form;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Defines an export display builder instance form.
 */
final class ExportForm extends FormBase {

  use AutowireTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $instance_id = NULL): array {
    $storage = $this->entityTypeManager->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->load($instance_id);

    if ($instance === NULL) {
      $this->messenger()->addError($this->t('Invalid instance id: @id', ['@id' => $instance_id]));
      $form['cancel'] = [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => new Url('entity.display_builder_instance.collection'),
      ];

      return $form;
    }

    $data = $instance->getCurrentState();
    self::cleanSourceId($data);

    try {
      $data = Yaml::encode($data);
    }
    catch (\Throwable $th) {
      $this->messenger()->addError($this->t('Failed to decode sources: @message', ['@message' => $th->getMessage()]));

      return $form;
    }

    $form['data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Export data'),
      '#rows' => '20',
      '#default_value' => $data,
    ];

    $form['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => new Url('entity.display_builder_instance.collection'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'display_builder_dev_tools_reset';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $form_state->setRedirectUrl(
      new Url(
        'entity.display_builder_instance.collection',
      )
    );
  }

  /**
   * Recursively regenerate the _source_id key.
   *
   * @param array $array
   *   The array reference.
   */
  private static function cleanSourceId(array &$array): void {
    unset($array['_source_id']);

    foreach ($array as $key => &$value) {
      if (\is_array($value)) {
        self::cleanSourceId($value);

        if (isset($value['source_id'], $value['source']['value']) && empty($value['source']['value'])) {
          unset($array[$key]);
        }
      }

      if ($value instanceof MarkupInterface) {
        $array[$key] = (string) $value;
      }
    }
  }

}
