<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\display_builder\InstanceInterface;
use Drupal\display_builder_dev_tools\MockEntity;

/**
 * Hook implementations for display_builder_dev_tools.
 */
class DisplayBuilderDevToolsHooks {

  /**
   * Implements hook_entity_operation_alter().
   *
   * @param array $operations
   *   An associative array of operations.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which the operations are being altered.
   */
  #[Hook('entity_operation_alter')]
  public function entityOperationAlter(array &$operations, EntityInterface $entity): void {
    if (!$entity instanceof InstanceInterface) {
      return;
    }

    $param = [
      'instance_id' => (string) $entity->id(),
    ];

    if (\str_starts_with((string) $entity->id(), MockEntity::getPrefix())) {
      $operations['build'] = [
        'title' => new TranslatableMarkup('Build display'),
        'url' => MockEntity::getUrlFromInstanceId((string) $entity->id()),
        'weight' => -1,
      ];
      $operations['edit'] = [
        'title' => new TranslatableMarkup('Edit'),
        'url' => Url::fromRoute('display_builder_dev_tools.edit', $param),
        'weight' => 10,
      ];
    }

    $operations['import'] = [
      'title' => new TranslatableMarkup('Import'),
      'url' => Url::fromRoute('display_builder_dev_tools.import', $param),
      'weight' => 20,
    ];

    $operations['export'] = [
      'title' => new TranslatableMarkup('Export'),
      'url' => Url::fromRoute('display_builder_dev_tools.export', $param),
      'weight' => 30,
    ];

    // Small helper to delete sample in preview. Sample generated is static.
    if (\str_starts_with((string) $entity->id(), 'entity_view__')) {
      [, $entity_type_id, $bundle] = \explode('__', $param['instance_id']);
      $operations['refresh_sample'] = [
        'title' => new TranslatableMarkup('Refresh sample'),
        'url' => Url::fromRoute('display_builder_dev_tools.delete_sample', [
          'entity_type_id' => $entity_type_id,
          'bundle' => $bundle,
        ]),
        'weight' => 90,
      ];
    }

    $operations['delete'] = [
      'title' => new TranslatableMarkup('Delete'),
      'url' => Url::fromRoute('display_builder_dev_tools.delete', $param),
      'weight' => 90,
    ];
  }

  /**
   * Implements hook_display_builder_provider_info().
   *
   * @return array
   *   An associative array of display builder providers.
   */
  #[Hook('display_builder_provider_info')]
  public function displayBuilderProviderInfo(): array {
    return [
      'devel' => [
        'label' => new TranslatableMarkup('Dev tools'),
        'class' => MockEntity::class,
        'context_requirement' => MockEntity::getContextRequirement(),
        'prefix' => MockEntity::getPrefix(),
      ],
    ];
  }

}
