<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools;

use Drupal\Core\Url;
use Drupal\display_builder\DisplayBuildableInterface;
use Drupal\display_builder\InstanceInterface;
use Drupal\display_builder\ProfileInterface;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * A class implementing DisplayBuildableInterface for standalone instances.
 */
final class MockEntity implements DisplayBuildableInterface {

  /**
   * A display builder instance.
   */
  private ?InstanceInterface $instance;

  /**
   * MockEntity constructor.
   *
   * @param string $instance_id
   *   The display builder instance ID.
   */
  public function __construct(string $instance_id) {
    $storage = \Drupal::service('entity_type.manager')->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->load($instance_id);
    $this->instance = $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function getPrefix(): string {
    return 'devel__';
  }

  /**
   * {@inheritdoc}
   */
  public static function getContextRequirement(): string {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public static function checkInstanceId(string $instance_id): ?array {
    return [
      'id' => $instance_id,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function createFromRoute(string $route, ParameterBag $params): ?DisplayBuildableInterface {
    if ($route !== 'display_builder_dev_tools.view') {
      return NULL;
    }
    return new MockEntity($params->get('instance_id'));
  }

  /**
   * {@inheritdoc}
   */
  public function getBuilderUrl(): Url {
    return Url::fromRoute('display_builder_dev_tools.view', ['instance_id' => $this->getInstanceId()]);
  }

  /**
   * {@inheritdoc}
   */
  public static function getUrlFromInstanceId(string $instance_id): Url {
    return Url::fromRoute('display_builder_dev_tools.view', ['instance_id' => $instance_id]);
  }

  /**
   * {@inheritdoc}
   *
   * phpcs:disable Generic.CodeAnalysis.UnusedFunctionParameter.FoundInImplementedInterface
   */
  public static function getDisplayUrlFromInstanceId(string $instance_id): Url {
    return Url::fromRoute('entity.display_builder_instance.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getProfile(): ?ProfileInterface {
    return $this->instance?->getProfile();
  }

  /**
   * {@inheritdoc}
   */
  public function getInstanceId(): ?string {
    return $this->instance ? (string) $this->instance->id() : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function initInstanceIfMissing(): void {}

  /**
   * {@inheritdoc}
   */
  public function getInitialSources(): array {
    return $this->getSources();
  }

  /**
   * {@inheritdoc}
   */
  public function getInitialContext(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getSources(): array {
    return $this->instance->getCurrentState();
  }

  /**
   * {@inheritdoc}
   */
  public function saveSources(): void {}

}
