<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\display_builder\ConfigFormBuilderInterface;
use Drupal\display_builder\DisplayBuilderHelpers;
use Drupal\display_builder_dev_tools\FixturesHelpers;
use Drupal\display_builder_dev_tools\MockEntity;

/**
 * Defines an add display builder instance form.
 */
final class AddForm extends FormBase {

  use AutowireTrait;

  public function __construct(
    protected ConfigFormBuilderInterface $configFormBuilder,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $display_builder = new MockEntity('');
    $form[ConfigFormBuilderInterface::PROFILE_PROPERTY] = $this->configFormBuilder->build($display_builder);

    $form['fixture_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Initial data'),
      '#description' => $this->t('Enter the fixture to use as base for this display builder instance.'),
      '#options' => FixturesHelpers::getAllFixturesOptions(),
      '#default_value' => 'blank',
      '#required' => TRUE,
    ];

    $form['instance_id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Builder ID'),
      '#size' => 20,
      '#maxlength' => 20,
      '#machine_name' => [
        'exists' => '\Drupal\display_builder\Entity\Instance::load',
      ],
      '#field_prefix' => MockEntity::getPrefix(),
      '#description' => $this->t('Uniq identifier for this instance, prefix is automatic to avoid conflict with regular display.'),
      '#default_value' => \uniqid(),
      '#required' => TRUE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => new Url('entity.display_builder_instance.collection'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'display_builder_dev_tools_add';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $id = \sprintf('%s%s', MockEntity::getPrefix(), $form_state->getValue('instance_id'));
    $fixture_id = $form_state->getValue('fixture_id', 'none:blank');
    [$extension_name, $fixture_id] = \explode(':', $fixture_id);
    $sources = DisplayBuilderHelpers::getFixtureDataFromExtension($extension_name, $fixture_id);

    $storage = $this->entityTypeManager->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface $instance */
    $instance = $storage->create([
      'id' => $id,
      'profileId' => $form_state->getValue(ConfigFormBuilderInterface::PROFILE_PROPERTY),
    ]);
    $instance->setNewPresent($sources, new TranslatableMarkup('Devel created with data: %fixture_id.', [
      '%fixture_id' => $fixture_id,
    ]));
    // Dev instance start with 1 history, need to clear for tests.
    // @todo find why this history is created.
    $instance->clear();
    $instance->save();

    $url = Url::fromRoute('display_builder_dev_tools.view', ['instance_id' => $instance->id()]);
    $form_state->setRedirectUrl($url);
  }

}
