<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

/**
 * Confirmation form to confirm deletion of display builder instance by id.
 */
class ConfirmDeleteForm extends ConfirmFormBase {

  use AutowireTrait;

  /**
   * Display builder id to delete.
   */
  private ?string $instanceId;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $instance_id = NULL): array {
    $this->instanceId = $instance_id;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('entity.display_builder_instance.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'confirm_display_builder_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Do you want to delete %id?', ['%id' => $this->instanceId]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $storage = \Drupal::service('entity_type.manager')->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->load($this->instanceId);
    $storage->delete([$instance]);
    $form_state->setRedirectUrl(new Url('entity.display_builder_instance.collection'));
  }

}
