<?php

declare(strict_types=1);

namespace Drupal\display_builder_dev_tools\Form;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\display_builder\DisplayBuilderHelpers;
use Drupal\display_builder\Event\DisplayBuilderEvent;
use Drupal\display_builder\Event\DisplayBuilderEvents;
use Drupal\display_builder_dev_tools\FixturesHelpers;
use Drupal\display_builder_dev_tools\MockEntity;
use Drupal\display_builder_entity_view\Entity\EntityViewDisplay;
use Drupal\display_builder_entity_view\Field\DisplayBuilderItemList;
use Drupal\display_builder_page_layout\Entity\PageLayout;
use Drupal\display_builder_views\Plugin\views\display_extender\DisplayExtender;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Defines an add display builder instance form.
 */
final class ImportForm extends FormBase {

  use AutowireTrait;

  /**
   * Mapping of context classes to human-readable labels.
   */
  protected array $contextClasses = [
    DisplayExtender::class,
    PageLayout::class,
    EntityViewDisplay::class,
    DisplayBuilderItemList::class,
  ];

  public function __construct(
    private readonly EventDispatcherInterface $eventDispatcher,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $instance_id = NULL): array {
    $storage = $this->entityTypeManager->getStorage('display_builder_instance');

    if ($storage->load($instance_id) === NULL) {
      $this->messenger()->addError($this->t('Invalid instance id: @id', ['@id' => $instance_id]));
      $form['cancel'] = [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => new Url('entity.display_builder_instance.collection'),
      ];

      return $form;
    }

    $options = FixturesHelpers::getAllFixturesOptions();

    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Import type'),
      '#options' => ['fixture' => $this->t('Fixture'), 'input' => $this->t('Input')],
      '#default_value' => 'fixture',
      '#required' => TRUE,
    ];

    $form['fixture_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Replace data'),
      '#description' => $this->t('Enter the fixture to replace in this display builder instance.'),
      '#options' => $options,
      '#default_value' => 'blank',
      '#states' => [
        'visible' => [
          ':input[name="type"]' => ['value' => 'fixture'],
        ],
      ],
    ];

    $form['instance_id'] = [
      '#type' => 'hidden',
      '#value' => $instance_id,
    ];

    $form['data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Import data'),
      '#description' => $this->t('Paste Yaml formatted data to import.'),
      '#states' => [
        'visible' => [
          ':input[name="type"]' => ['value' => 'input'],
        ],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $form['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => new Url('entity.display_builder_instance.collection'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'display_builder_dev_tools_reset';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $import_type = $form_state->getValue('type', 'fixture');

    $instance_id = $form_state->getValue('instance_id');

    if ($import_type === 'fixture') {
      $fixture_id = $form_state->getValue('fixture_id', 'none:blank');
      [$extension_name, $fixture_id] = \explode(':', $fixture_id);
      $sources = DisplayBuilderHelpers::getFixtureDataFromExtension($extension_name, $fixture_id);
    }
    else {
      $sources = Yaml::decode($form_state->getValue('data'));
    }

    $storage = $this->entityTypeManager->getStorage('display_builder_instance');
    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->load($instance_id);

    if (!$instance) {
      $this->messenger()->addError($this->t('The display builder instance %id does not exist.', ['%id' => $instance_id]));

      return;
    }

    $contexts = $instance->getContexts();
    $storage->delete([$instance]);

    /** @var \Drupal\display_builder\InstanceInterface|null $instance */
    $instance = $storage->create([
      'id' => $instance_id,
      'profileId' => (string) $instance->getProfile()->id(),
      'contexts' => $contexts,
    ]);
    $instance->setNewPresent($sources, 'Import');
    $instance->save();

    $url = NULL;

    foreach ($this->contextClasses as $class) {
      if (\class_exists($class) && $instance->hasSaveContextsRequirement($class::getContextRequirement(), $contexts ?? [])) {
        $url = $class::getUrlFromInstanceId($instance_id);
        // Save is different for each target, we simulate save by triggering the
        // DisplayBuilderEvents::ON_SAVE.
        $event = new DisplayBuilderEvent($instance_id, [], [], $contexts);
        $this->eventDispatcher->dispatch($event, DisplayBuilderEvents::ON_SAVE);

        break;
      }

      $url = MockEntity::getUrlFromInstanceId($instance_id);
    }

    $form_state->setRedirectUrl($url);
    // phpcs:ignore
    \Drupal::service('plugin.cache_clearer')->clearCachedDefinitions();
  }

}
