/**
 * @file
 * Countdown functionality for the default Drupal Idle Timer modal.
 */

(function (Drupal) {
  'use strict';

  Drupal.behaviors.drupalIdleTimerModalCountdown = {
    attach: function (context, settings) {
      if (!settings.drupalIdleTimer) return;

      const { classes, timeout, warningDuration } = settings.drupalIdleTimer;
      const modal = document.getElementById(classes.modal);

      if (!modal || modal.dataset.countdownAttached) return;

      modal.dataset.countdownAttached = 'true';

      const countdownElement = modal.querySelector('[data-countdown-time]');
      if (!countdownElement) return;

      let countdownInterval = null;
      let remainingSeconds = Math.floor(warningDuration / 1000);

      // Function to update countdown display
      function updateCountdown() {
        if (countdownElement) {
          countdownElement.textContent = remainingSeconds;
        }

        if (remainingSeconds > 0) {
          remainingSeconds--;
        } else {
          clearInterval(countdownInterval);
        }
      }

      // Start countdown when modal opens
      const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
          if (mutation.attributeName === 'class') {
            const modalOpenClasses = classes.modalOpen.split(' ');
            const hasOpenClass = modalOpenClasses.some(cls => modal.classList.contains(cls));

            if (hasOpenClass) {
              // Reset and start countdown
              remainingSeconds = Math.floor(warningDuration / 1000);
              updateCountdown();

              if (countdownInterval) {
                clearInterval(countdownInterval);
              }

              countdownInterval = setInterval(updateCountdown, 1000);
            } else {
              // Modal closed, clear countdown
              if (countdownInterval) {
                clearInterval(countdownInterval);
                countdownInterval = null;
              }
              remainingSeconds = Math.floor(warningDuration / 1000);
              if (countdownElement) {
                countdownElement.textContent = remainingSeconds;
              }
            }
          }
        });
      });

      observer.observe(modal, {
        attributes: true,
        attributeFilter: ['class']
      });
    }
  };

})(Drupal);
