<?php

namespace Drupal\drupal_idle_timer\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AnonymousUserSession;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * Controller for the Idle Timer.
 */
class IdleTimerController extends ControllerBase {

  /**
   * Constructs a new IdleTimerController.
   */
  public function __construct(protected Request $request, protected Session $session) {}

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('session')
    );
  }

  /**
   * AJAX keepalive: update last active timestamp in session.
   */
  public function keepalive() {
    $session = $this->request->getSession();
    if ($session) {
      $session->set('drupal_idle_timer_last_active', time());
    }

    return new JsonResponse([
      'status' => 'ok',
      'timestamp' => time(),
    ]);
  }

  /**
   * Return last active timestamp from session.
   */
  public function getLastActive() {
    $session = $this->request->getSession();
    $lastActive = $session->get('drupal_idle_timer_last_active', time());

    return new JsonResponse([
      'lastActive' => $lastActive * 1000,
      'serverTime' => time() * 1000,
    ]);
  }

  /**
   * AJAX logout: destroy session and return AjaxResponse.
   */
  public function logout() {
    $user = $this->currentUser();

    // Log the idle logout event before logging out.
    if ($user->isAuthenticated()) {
      $this->logIdleLogout($user);
    }

    $this->session->clear();
    $user->setAccount(new AnonymousUserSession());

    $response = new AjaxResponse();
    $response->setStatusCode(200);

    return $response;
  }

  /**
   * Log an idle logout event to the database.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account being logged out.
   */
  protected function logIdleLogout($account) {
    $connection = \Drupal::database();

    try {
      $connection->insert('drupal_idle_timer_logs')
        ->fields([
          'uid' => $account->id(),
          'username' => $account->getAccountName(),
          'logout_time' => time(),
          'ip_address' => $this->request->getClientIp(),
          'user_agent' => $this->request->headers->get('User-Agent'),
        ])
        ->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('drupal_idle_timer')->error('Failed to log idle logout: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

}
