<?php

namespace Drupal\drupal_idle_timer\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Idle Timer reports.
 */
class IdleTimerReportsController extends ControllerBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new IdleTimerReportsController.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(DateFormatterInterface $date_formatter, Connection $database, EntityTypeManagerInterface $entity_type_manager, RequestStack $request_stack) {
    $this->dateFormatter = $date_formatter;
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('date.formatter'),
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Display idle logout logs.
   *
   * @return array
   *   A render array containing the idle logout logs table.
   */
  public function reports() {
    $build = [];

    // Get configuration.
    $config = \Drupal::config('drupal_idle_timer.settings');
    $retention_days = $config->get('log_retention_days') ?? 90;

    $description_text = $this->t('This report shows all instances where users were automatically logged out due to inactivity.');
    if ($retention_days > 0) {
      $description_text .= ' ' . $this->t('Logs are automatically deleted after @days days.', ['@days' => $retention_days]);
    }
    else {
      $description_text .= ' ' . $this->t('Logs are kept indefinitely.');
    }

    $build['description'] = [
      '#markup' => '<p>' . $description_text . '</p>',
    ];

    // Get filter parameters from query string.
    $request = $this->requestStack->getCurrentRequest();
    $username_filter = $request->query->get('username');
    $date_from = $request->query->get('date_from');
    $date_to = $request->query->get('date_to');

    // Add filter form.
    $build['filters'] = $this->buildFilterForm($username_filter, $date_from, $date_to);

    // Build the logs table.
    $build['logs_table'] = $this->buildLogsTable($username_filter, $date_from, $date_to);

    return $build;
  }

  /**
   * Build the filter form.
   *
   * @param string|null $username_filter
   *   The username filter value.
   * @param string|null $date_from
   *   The date from filter value.
   * @param string|null $date_to
   *   The date to filter value.
   *
   * @return array
   *   A render array for the filter form.
   */
  protected function buildFilterForm($username_filter, $date_from, $date_to) {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['idle-timer-filters']],
      'form' => [
        '#type' => 'html_tag',
        '#tag' => 'form',
        '#attributes' => ['method' => 'get', 'class' => ['form-inline']],
        'username' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['form-item']],
          'label' => [
            '#type' => 'html_tag',
            '#tag' => 'label',
            '#value' => $this->t('Username:'),
          ],
          'input' => [
            '#type' => 'html_tag',
            '#tag' => 'input',
            '#attributes' => [
              'type' => 'text',
              'name' => 'username',
              'value' => $username_filter ?: '',
              'placeholder' => $this->t('Filter by username'),
            ],
          ],
        ],
        'date_from' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['form-item']],
          'label' => [
            '#type' => 'html_tag',
            '#tag' => 'label',
            '#value' => $this->t('From:'),
          ],
          'input' => [
            '#type' => 'html_tag',
            '#tag' => 'input',
            '#attributes' => [
              'type' => 'date',
              'name' => 'date_from',
              'value' => $date_from ?: '',
            ],
          ],
        ],
        'date_to' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['form-item']],
          'label' => [
            '#type' => 'html_tag',
            '#tag' => 'label',
            '#value' => $this->t('To:'),
          ],
          'input' => [
            '#type' => 'html_tag',
            '#tag' => 'input',
            '#attributes' => [
              'type' => 'date',
              'name' => 'date_to',
              'value' => $date_to ?: '',
            ],
          ],
        ],
        'submit' => [
          '#type' => 'html_tag',
          '#tag' => 'button',
          '#attributes' => [
            'type' => 'submit',
            'class' => ['button', 'button--primary'],
          ],
          '#value' => $this->t('Filter'),
        ],
        'reset' => [
          '#type' => 'html_tag',
          '#tag' => 'a',
          '#attributes' => [
            'href' => Url::fromRoute('drupal_idle_timer.reports')->toString(),
            'class' => ['button'],
          ],
          '#value' => $this->t('Reset'),
        ],
      ],
    ];
  }

  /**
   * Build the logs table.
   *
   * @param string|null $username_filter
   *   The username filter value.
   * @param string|null $date_from
   *   The date from filter value.
   * @param string|null $date_to
   *   The date to filter value.
   *
   * @return array
   *   A render array for the logs table.
   */
  protected function buildLogsTable($username_filter, $date_from, $date_to) {
    // Build query.
    $query = $this->database->select('drupal_idle_timer_logs', 'l')
      ->fields('l')
      ->orderBy('logout_time', 'DESC');

    // Apply filters.
    if (!empty($username_filter)) {
      $query->condition('username', '%' . $this->database->escapeLike($username_filter) . '%', 'LIKE');
    }

    if (!empty($date_from)) {
      $timestamp_from = strtotime($date_from . ' 00:00:00');
      if ($timestamp_from !== FALSE) {
        $query->condition('logout_time', $timestamp_from, '>=');
      }
    }

    if (!empty($date_to)) {
      $timestamp_to = strtotime($date_to . ' 23:59:59');
      if ($timestamp_to !== FALSE) {
        $query->condition('logout_time', $timestamp_to, '<=');
      }
    }

    // Add pager.
    $query = $query->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(50);

    $results = $query->execute()->fetchAll();

    // Build table rows.
    $rows = [];
    foreach ($results as $record) {
      $user_link = $this->t('N/A');
      if ($record->uid > 0) {
        $user = $this->entityTypeManager->getStorage('user')->load($record->uid);
        if ($user) {
          $user_link = Link::createFromRoute(
            $record->username,
            'entity.user.canonical',
            ['user' => $record->uid]
          );
        }
        else {
          $user_link = $record->username . ' ' . $this->t('(deleted)');
        }
      }

      $rows[] = [
        $user_link,
        $this->dateFormatter->format($record->logout_time, 'short'),
        $record->ip_address ?: $this->t('Unknown'),
        !empty($record->user_agent) ? $this->truncateString($record->user_agent, 100) : $this->t('Unknown'),
      ];
    }

    $build = [
      '#theme' => 'table',
      '#header' => [
        $this->t('User'),
        $this->t('Logout Time'),
        $this->t('IP Address'),
        $this->t('User Agent'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No idle logout events found.'),
    ];

    // Add pager.
    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Truncate a string to a specified length.
   *
   * @param string $string
   *   The string to truncate.
   * @param int $length
   *   The maximum length.
   *
   * @return string
   *   The truncated string.
   */
  protected function truncateString($string, $length) {
    if (strlen($string) > $length) {
      return substr($string, 0, $length) . '...';
    }
    return $string;
  }

  /**
   * Page title callback.
   *
   * @return string
   *   The page title.
   */
  public function title() {
    return $this->t('Idle Logout Reports');
  }

}
