(function (Drupal, $) {
  function debounce(fn, wait) {
    let t;
    return function debounced(...args) {
      clearTimeout(t);
      t = setTimeout(() => fn(...args), wait);
    };
  }

  Drupal.behaviors.drupalIdleTimer = {
    attach(context, settings) {
      if (!settings.drupalIdleTimer) return;
      if (
        context !== document &&
        !(context instanceof Element && context.matches('body'))
      )
        return;

      const {
        syncMethod,
        timeout,
        warningDuration,
        pollInterval,
        debounce: debounceInterval,
        events,
        debug,
        classes,
        logoutUrl,
        logoutRedirectUrl,
      } = settings.drupalIdleTimer;
      const keepAliveUrl = Drupal.url('drupal-idle-timer/keepalive');
      const statusUrl = Drupal.url('drupal-idle-timer/status');

      let idle = false;
      let logoutTimer = null;

      function logDebug(msg, variable = null) {
        if (debug) {
          console.log(msg, variable);
        }
      }

      function getRemainingTime(serverLastActive, serverNow) {
        const elapsed = serverNow - serverLastActive;
        return warningDuration - elapsed;
      }

      function hideModal() {
        const $dialog = $(`#${classes.modal}`);
        $dialog.removeClass(classes.modalOpen);
        if (logoutTimer) {
          clearTimeout(logoutTimer);
          logoutTimer = null;
        }
      }

      function triggerLogoutEvent(logoutMethod) {
        logDebug('[IdleTimer] Triggering logout event:', logoutMethod);
        hideModal();
        idle = true;

        const logoutEvent = new CustomEvent(logoutMethod, {
          detail: {
            logoutMethod,
            logoutUrl,
          },
        });
        document.dispatchEvent(logoutEvent);
        logDebug('[IdleTimer] Logout event dispatched:', logoutMethod);

        if (logoutUrl) {
          // Custom logout URL - redirect there directly
          window.location = logoutUrl;
        } else {
          // Use default logout endpoint
          $.getJSON({
            url: Drupal.url('idle-logout'),
            type: 'POST',
            dataType: 'json',
            success(response) {
              logDebug('[IdleTimer] Logout successful:', response);

              // Make sure everything is properly cleaned up.
              if (syncMethod === 'localstorage') {
                localStorage.removeItem('idleTimer');
              } else if (syncMethod === 'sessionstorage') {
                sessionStorage.removeItem('idleTimer');
              }

              // Redirect to the configured logout redirect URL
              const redirectUrl =
                response.redirect_url || logoutRedirectUrl || '/';
              window.location = redirectUrl;
            },
            error() {
              // If logout fails, still redirect to the configured page
              logDebug('[IdleTimer] Logout error, redirecting anyway');
              window.location = logoutRedirectUrl || '/';
            },
          });
        }
      }

      function setSession() {
        logDebug('[IdleTimer] Sending keepalive...');

        if (syncMethod === 'localstorage') {
          localStorage.setItem('idleTimer', Date.now().toString());
          hideModal();
          idle = false;
        } else if (syncMethod === 'sessionstorage') {
          sessionStorage.setItem('idleTimer', Date.now().toString());
          hideModal();
          idle = false;
        } else {
          $.getJSON(keepAliveUrl).done(() => {
            hideModal();
            idle = false;
          });
        }

        logDebug('[IdleTimer] Keepalive successful.');
      }

      function showModal() {
        const $dialog = $(`#${classes.modal}`);
        if ($dialog.hasClass(classes.modalOpen)) return;

        $dialog.addClass(classes.modalOpen);

        // hook up buttons once
        const $buttons = document.querySelectorAll(classes.buttonAttributes);
        $buttons.forEach(($btn) => {
          $btn.addEventListener(
            'click',
            () => {
              if (classes.logout && $btn.classList.contains(classes.logout)) {
                triggerLogoutEvent('normal');
              } else if (
                classes.staySignedIn &&
                $btn.classList.contains(classes.staySignedIn)
              ) {
                setSession();
              }
              clearTimeout(logoutTimer);
            },
            { once: true },
          );
        });

        // auto-logout when countdown expires
        logoutTimer = setTimeout(
          () => triggerLogoutEvent('idle'),
          timeout - warningDuration,
        );
      }

      function checkModal(remaining) {
        if (remaining <= 0 && !idle) {
          idle = true;
          showModal();
        } else if (remaining > 0 && idle) {
          idle = false;
          hideModal();
        }
      }

      function checkIdle() {
        if (syncMethod === 'localstorage') {
          const lastActive = parseInt(
            localStorage.getItem('idleTimer') || Date.now(),
            10,
          );

          const now = Date.now();
          const remaining = getRemainingTime(lastActive, now);

          logDebug('[IdleTimer] Checking localStorage. Remaining:', remaining);
          checkModal(remaining);
        } else if (syncMethod === 'ajax') {
          $.getJSON(statusUrl).done((data) => {
            const remaining = getRemainingTime(
              data.lastActive || Date.now(),
              data.serverTime || Date.now(),
            );

            logDebug('[IdleTimer] Checking AJAX. Remaining:', remaining);
            checkModal(remaining);
          });
        } else if (syncMethod === 'sessionstorage') {
          const lastActive = parseInt(
            sessionStorage.getItem('idleTimer') || Date.now(),
            10,
          );
          const now = Date.now();
          const remaining = getRemainingTime(lastActive, now);

          logDebug(
            '[IdleTimer] Checking sessionStorage. Remaining:',
            remaining,
          );
          checkModal(remaining);
        }
      }

      // Start polling.
      setInterval(checkIdle, pollInterval);

      // User actions = keep alive.
      const actionHandler = debounce(() => {
        if (syncMethod === 'localstorage') {
          localStorage.setItem('idleTimer', Date.now().toString());
        }
        if (syncMethod === 'sessionstorage') {
          sessionStorage.setItem('idleTimer', Date.now().toString());
        }
        if (syncMethod === 'ajax') {
          $.getJSON(keepAliveUrl);
        }
      }, debounceInterval || 250);
      (events || ['mousemove', 'keydown', 'mousedown', 'touchstart']).forEach(
        (ev) => document.addEventListener(ev, actionHandler, true),
      );

      setSession();
    },
  };
})(Drupal, jQuery);
