<?php

namespace Drupal\drupal_idle_timer\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AnonymousUserSession;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * Controller for the Idle Timer.
 */
class IdleTimerController extends ControllerBase {

  /**
   * Constructs a new IdleTimerController.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Symfony\Component\HttpFoundation\Session\Session $session
   *   The session service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected Request $request,
    protected Session $session,
    protected $configFactory,
    protected Connection $database,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('session'),
      $container->get('config.factory'),
      $container->get('database'),
      $container->get('logger.factory')->get('drupal_idle_timer')
    );
  }

  /**
   * AJAX keepalive: update last active timestamp in session.
   */
  public function keepalive() {
    $session = $this->request->getSession();
    if ($session) {
      $session->set('drupal_idle_timer_last_active', time());
    }

    return new JsonResponse([
      'status' => 'ok',
      'timestamp' => time(),
    ]);
  }

  /**
   * Return last active timestamp from session.
   */
  public function getLastActive() {
    $session = $this->request->getSession();
    $lastActive = $session->get('drupal_idle_timer_last_active', time());

    return new JsonResponse([
      'lastActive' => $lastActive * 1000,
      'serverTime' => time() * 1000,
    ]);
  }

  /**
   * AJAX logout: destroy session and return redirect URL.
   */
  public function logout() {
    $user = $this->currentUser();

    // Log the idle logout event before logging out.
    if ($user->isAuthenticated()) {
      $this->logIdleLogout($user);
    }

    $this->session->clear();
    $user->setAccount(new AnonymousUserSession());

    // Get the configured redirect URL.
    $config = $this->configFactory->get('drupal_idle_timer.settings');
    $redirect_url = $config->get('logout_redirect_url');

    // Default to front page if no redirect URL is configured.
    if (empty($redirect_url)) {
      $redirect_url = '/';
    }

    return new JsonResponse([
      'status' => 'ok',
      'redirect_url' => $redirect_url,
    ]);
  }

  /**
   * Log an idle logout event to the database.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account being logged out.
   */
  protected function logIdleLogout($account) {
    try {
      $this->database->insert('drupal_idle_timer_logs')
        ->fields([
          'uid' => $account->id(),
          'username' => $account->getAccountName(),
          'logout_time' => time(),
          'ip_address' => $this->request->getClientIp(),
          'user_agent' => $this->request->headers->get('User-Agent'),
        ])
        ->execute();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to log idle logout: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

}
