<?php

namespace Drupal\drupal_idle_timer\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The IdleTimerSettingsForm class.
 */
class IdleTimerSettingsForm extends ConfigFormBase {

  /**
   * The IdleTimerSettingsForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface|null $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['drupal_idle_timer.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'drupal_idle_timer_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('drupal_idle_timer.settings');

    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Configure the idle timer settings.'),
    ];

    // Get all user roles.
    $roles = $this->entityTypeManager
      ->getStorage('user_role')
      ->loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      $role_options[$role->id()] = $role->label();
    }

    // Add the select field for roles.
    $form['allowed_roles'] = [
      '#type' => 'select',
      '#title' => $this->t('Allowed roles'),
      '#description' => $this->t('Select which roles this setting applies to.'),
      '#options' => $role_options,
      '#default_value' => $config->get('allowed_roles') ?? [],
      '#multiple' => TRUE,
    ];

    $form['sync_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Idle sync method'),
      '#description' => $this->t('Choose how the idle timer should synchronize between tabs. Defaults to AJAX polling.'),
      '#options' => [
        'ajax' => $this->t('AJAX polling'),
        'localstorage' => $this->t('Local Storage'),
        'sessionstorage' => $this->t('Session Storage'),
      ],
      '#default_value' => $config->get('sync_method') ?? 'ajax',
    ];

    $form['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Session timeout (ms)'),
      '#description' => $this->t('The maximum duration of user inactivity before automatic logout. Defaults to 900000 ms.'),
      '#default_value' => $config->get('timeout') ?? 900000,
      '#min' => 1000,
    ];

    $form['debounce'] = [
      '#type' => 'number',
      '#title' => $this->t('Debounce (ms)'),
      '#default_value' => $config->get('debounce') ?? 250,
      '#description' => $this->t('The debounce time for user activity events. Defaults to 250 ms.'),
      '#min' => 50,
    ];

    $form['warning_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Warning duration (ms)'),
      '#description' => $this->t('The duration for which the warning modal is displayed before automatic logout. Defaults to 60000 ms.'),
      '#default_value' => $config->get('warning_duration') ?? 60000,
      '#min' => 1000,
    ];

    $form['poll_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Poll interval (ms)'),
      '#description' => $this->t('The interval at which the server is polled for user activity. Defaults to 5000 ms.'),
      '#default_value' => $config->get('poll_interval') ?? 5000,
      '#min' => 1000,
    ];

    $form['logout_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Logout URL'),
      '#description' => $this->t('Only use if you have a custom logout URL. The default uses the Drupal logout route.'),
      '#default_value' => $config->get('logout_url') ?? '',
    ];

    $form['logout_redirect_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Logout redirect URL'),
      '#description' => $this->t('The page to redirect users to after they are logged out due to inactivity. Can be an internal path (e.g., "/user/login") or external URL (e.g., "https://example.com"). Leave empty to redirect to the front page.'),
      '#default_value' => $config->get('logout_redirect_url') ?? '',
      '#placeholder' => $this->t('/user/login'),
    ];

    $form['modal_identifier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS identifier for modal'),
      '#description' => $this->t('Identifier to apply to the modal element. Defaults to "activity-timeout".'),
      '#default_value' => $config->get('modal_identifier') ?? 'activity-timeout',
    ];

    $form['class_modal_open'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS class(es) for open modal'),
      '#description' => $this->t('Space-separated list of CSS classes to apply to the modal when it is open.'),
      '#default_value' => $config->get('class_modal_open') ?? 'is-open m-modal--is-open',
    ];

    $form['button_attributes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button attributes for action buttons'),
      '#description' => $this->t('Array of attributes to apply to the action buttons from the modal. Defaults to "[data-modal-dismiss]". Make sure to include the brackets.'),
      '#default_value' => $config->get('button_attributes') ?? '[data-modal-dismiss]',
    ];

    $form['class_logout'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS class(es) for logout button'),
      '#description' => $this->t('Space-separated list of CSS classes to apply to the logout button from the modal. Defaults to "submit-timeout".'),
      '#default_value' => $config->get('class_logout') ?? 'submit-timeout',
    ];

    $form['class_stay_signed_in'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS class(es) for stay signed in button'),
      '#description' => $this->t('Space-separated list of CSS classes to apply to the stay signed in button from the modal. Defaults to "stay-logged-in".'),
      '#default_value' => $config->get('class_stay_signed_in') ?? 'stay-logged-in',
    ];

    $form['use_default_modal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use default modal'),
      '#description' => $this->t('If enabled, a default modal will be used for the idle timeout warning. If disabled, you need to provide your own modal in your theme.'),
      '#default_value' => $config->get('use_default_modal') ?? TRUE,
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug logging'),
      '#description' => $this->t('WARNING: FOR DEVELOPMENT ONLY, USE WITH CAUTION!! If enabled, idle timer activity will be logged to the browser console.'),
      '#default_value' => $config->get('debug') ?? FALSE,
    ];

    $form['log_retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Log retention (days)'),
      '#description' => $this->t('Number of days to keep idle logout logs. Logs older than this will be automatically deleted during cron runs. Set to 0 to keep logs indefinitely.'),
      '#default_value' => $config->get('log_retention_days') ?? 90,
      '#min' => 0,
      '#field_suffix' => $this->t('days'),
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('drupal_idle_timer.settings')
      ->set('allowed_roles', $form_state->getValue('allowed_roles'))
      ->set('sync_method', $form_state->getValue('sync_method'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('debounce', $form_state->getValue('debounce'))
      ->set('warning_duration', $form_state->getValue('warning_duration'))
      ->set('poll_interval', $form_state->getValue('poll_interval'))
      ->set('logout_url', $form_state->getValue('logout_url'))
      ->set('logout_redirect_url', $form_state->getValue('logout_redirect_url'))
      ->set('modal_identifier', $form_state->getValue('modal_identifier'))
      ->set('class_modal_open', $form_state->getValue('class_modal_open'))
      ->set('button_attributes', $form_state->getValue('button_attributes'))
      ->set('class_logout', $form_state->getValue('class_logout'))
      ->set('class_stay_signed_in', $form_state->getValue('class_stay_signed_in'))
      ->set('use_default_modal', $form_state->getValue('use_default_modal'))
      ->set('debug', $form_state->getValue('debug'))
      ->set('log_retention_days', $form_state->getValue('log_retention_days'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
