<?php

namespace Drupal\diwali_email_card\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\diwali_email_card\Service\DiwaliEmailCardService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Diwali Card Email Form.
 */
class DiwaliEmailCardForm extends FormBase {

  /**
   * The Diwali card service.
   *
   * @var \Drupal\diwali_email_card\Service\DiwaliEmailCardService
   */
  protected $diwaliCardService;

  /**
   * Constructs a DiwaliEmailCardForm object.
   *
   * @param \Drupal\diwali_email_card\Service\DiwaliEmailCardService $diwali_card_service
   *   The Diwali card service.
   */
  public function __construct(DiwaliEmailCardService $diwali_card_service) {
    $this->diwaliCardService = $diwali_card_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('diwali_email_card.pdf_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'diwali_email_card_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['recipient_emails'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Recipient Emails'),
      '#description' => $this->t('Enter one or more email addresses separated by commas or new lines.'),
      '#required' => TRUE,
      '#attributes' => [
        'style' => 'width: 38rem; height: 100px;',
      ],
    ];

    $form['company_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name'),
      '#description' => $this->t('Enter the company or sender name to display on the card.'),
      '#required' => TRUE,
    ];

    $form['card_design'] = [
      '#type' => 'select',
      '#title' => $this->t('Choose Diwali Card Design'),
      '#options' => [
        'card1' => $this->t('Card Design 1'),
        'card2' => $this->t('Card Design 2'),
        'card3' => $this->t('Card Design 3'),
        'card4' => $this->t('Card Design 4'),
        'card5' => $this->t('Card Design 5'),
      ],
      '#required' => TRUE,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate and Send'),
    ];

    $form['pdf_preview'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'pdf-preview-wrapper'],
    ];

    $pdf_url = $form_state->get('pdf_file_url');
    if ($pdf_url) {
      $form['pdf_preview']['link'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<a href=":url" target="_blank" rel="noopener noreferrer">Download Generated PDF</a>', [':url' => $pdf_url]),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $emails_input = $form_state->getValue('recipient_emails');
    $emails = preg_split('/[\s,]+/', $emails_input, -1, PREG_SPLIT_NO_EMPTY);

    foreach ($emails as $email) {
      $email = trim($email);
      if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $form_state->setErrorByName('recipient_emails', $this->t('Invalid email address: %email', ['%email' => $email]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $card_design = $form_state->getValue('card_design');
    $company_name = $form_state->getValue('company_name');
    $emails_input = $form_state->getValue('recipient_emails');
    $emails = preg_split('/[\s,]+/', $emails_input, -1, PREG_SPLIT_NO_EMPTY);

    // Get service from container.
    $diwali_service = \Drupal::service('diwali_email_card.pdf_service');

    $pdf_path = $diwali_service->generatePdfCard($card_design, $company_name);

    if (!$pdf_path) {
      $this->messenger()->addError($this->t('Failed to generate Diwali card PDF.'));
      return;
    }
    // Loop over $emails to send emails or process further.
    foreach ($emails as $email) {
      $email = trim($email);
      if (!empty($email) && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $diwali_service->sendEmailWithAttachment($email, $pdf_path);
      }
      else {
        $this->messenger()->addWarning($this->t('Invalid email skipped: %email', ['%email' => $email]));
      }
    }

    $form_state->setRebuild(TRUE);
  }

}
