<?php

namespace Drupal\diwali_email_card\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service to generate Diwali card PDF and send email.
 */
class DiwaliEmailCardService {

  protected $mailManager;
  protected $configFactory;
  protected $currentUser;
  protected $messenger;
  protected $translation;
  protected $moduleExtensionList;

  /**
   * {@inheritdoc}
   */
  public function __construct(MailManagerInterface $mail_manager, ConfigFactoryInterface $config_factory, AccountInterface $current_user, MessengerInterface $messenger, TranslationInterface $translation, ModuleExtensionList $extension_list_module) {
    $this->mailManager = $mail_manager;
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
    $this->translation = $translation;
    $this->moduleExtensionList = $extension_list_module;
  }

  /**
   * Generate Diwali card PDF.
   *
   * @param string $card_design
   *   The card design name (e.g. 'card1', 'card2').
   *
   * @return string|false
   *   The full file path to saved PDF or false on failure.
   */
  public function generatePdfCard(string $card_design, string $company_name) {
    // Load template HTML for selected card design.
    $template_path = DRUPAL_ROOT . "/modules/contrib/diwali_email_card/templates/{$card_design}.html";
    if (!file_exists($template_path)) {
      $this->messenger->addError($this->translation->translate('Selected card design not found.'));
      return FALSE;
    }

    $html = file_get_contents($template_path);
    $html = str_replace('{{ company_name }}', htmlspecialchars($company_name, ENT_QUOTES, 'UTF-8'), $html);

    // Initialize TCPDF.
    $pdf = new \TCPDF();
    $pdf->SetCreator(PDF_CREATOR);
    $pdf->SetAuthor($this->configFactory->get('system.site')->get('name'));
    $pdf->SetTitle('Diwali Card');
    $pdf->SetMargins(15, 15, 15);
    $pdf->SetAutoPageBreak(TRUE, 15);
    $pdf->AddPage();
    $pdf->setFont('dejavusans', '', 12);
    $pdf->writeHTML($html, TRUE, FALSE, TRUE, FALSE, '');

    $filename = 'diwali_card_' . time() . '.pdf';
    $file_path = 'public://' . $filename;
    $pdf->Output(\Drupal::service('file_system')->realpath($file_path), 'F');

    return \Drupal::service('file_system')->realpath($file_path);
  }

  /**
   * Send Diwali card email with attachment.
   *
   * @param string $to
   *   Recipient email.
   * @param string $pdf_path
   *   Full file path of the PDF to attach.
   */
  public function sendEmailWithAttachment(string $to, string $pdf_path) {
    $module = 'diwali_email_card';
    $key = 'send_diwali_card';
    $langcode = $this->currentUser->getPreferredLangcode();
    $send = TRUE;

    if (!file_exists($pdf_path)) {
      $this->messenger->addError($this->translation->translate('PDF file not found, email not sent.'));
      return;
    }

    $params = [
      'subject' => $this->translation->translate('🎉 Happy Diwali Wishes!'),
      'message' => $this->translation->translate('<p>Please find your Diwali card attached.</p><p>Happy Diwali!</p>'),
      'attachments' => [
        [
          'filecontent' => file_get_contents($pdf_path),
          'filename' => basename($pdf_path),
          'filemime' => 'application/pdf',
        ],
      ],
    ];

    $result = $this->mailManager->mail($module, $key, $to, $langcode, $params, NULL, $send);

    if ($result['result'] !== TRUE) {
      $this->messenger->addError($this->translation->translate('Failed to send Diwali card email.'));
    }
    else {
      $this->messenger->addMessage($this->translation->translate('Diwali card sent to @email', ['@email' => $to]));
    }
  }

}
