<?php

namespace Drupal\dkan_dataset_archiver\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Alt Text Validation Setting form.
 */
class ArchiverSettings extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dkan_dataset_archiver_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dkan_dataset_archiver.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $form['#title'] = $this->t('DKAN Dataset Archiver Settings');

    $form['archive'] = [
      '#type' => 'radios',
      '#title' => $this->t('Archive datasets when updated, deleted, or unpublished.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('archive') ?? 0,
      '#options' => [
        '1' => $this->t('On'),
        '0' => $this->t('Off'),
      ],
    ];

    $form['collation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('How to group archived datasets'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['collation']['archive_by_theme'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset theme'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Theme when updated, deleted, or unpublished in the same 24 hour period.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('archive_by_theme') ?? FALSE,
    ];

    $form['collation']['archive_by_keyword'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset keyword'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Keywords when updated, deleted, or unpublished in the same 24 hour period.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('archive_by_keyword') ?? FALSE,
    ];

    $form['collation']['create_annual_archives'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create annual archives'),
      '#description' => $this->t('Select this if you want to create annual archives of all datasets. This will create a zip file of all datasets at the end of each year. It will create them keyword and theme if those are set.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('create_annual_archives') ?? FALSE,
    ];

    $form['collation']['include_manifest'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include manifest file in archives'),
      '#description' => $this->t('Select this if you want to include a manifest file (a text file listing the datasets included) in each archive created.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('include_manifest') ?? 0,
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
          [':input[name="create_annual_archives"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['archive_private'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive private datasets'),
      '#description' => $this->t('Select this if you want to create Theme, Keyword or Annual archives of private datasets. These would be separate archives and visible only by specific permission'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('archive_private') ?? 0,
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
          [':input[name="create_annual_archives"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['locations_group'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Archive Storage and Retention'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['storage_locations'] = [
      '#type' => 'radios',
      '#title' => $this->t('Storage location'),
      '#description' => $this->t('Choose where you would like to store the dataset archives.'),
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('storage_locations') ?? 0,
      '#options' => [
        'local' => $this->t('Create only local archives %example', ['%example' => '(files/dkan_dataset_archives/)']),
        'remote' => $this->t('Create only remote archives'),
        'local_and_remote' => $this->t('Create archives in both local and remote locations'),
      ],
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['local_years_retained'] = [
      '#prefix' => '<div class="container-inline">',
      '#title' => $this->t('Number of years to retain local archives'),
      '#description' => $this->t('Set to 0 to retain indefinitely.'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 3,
      '#max' => 365,
      '#min' => 0,
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('local_years_retained') ?? 0,
      '#suffix' => '</div>',
      '#states' => [
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'local'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    $form['locations_group']['remote_years_retained'] = [
      '#prefix' => '<div class="container-inline">',
      '#title' => $this->t('Number of years to retain remote archives'),
      '#description' => $this->t('Set to 0 to retain indefinitely.'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 3,
      '#max' => 365,
      '#min' => 0,
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('remote_years_retained') ?? 0,
      '#suffix' => '</div>',
      '#states' => [
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    $form['locations_group']['remote_address'] = [
      '#title' => $this->t('The AWS S3 bucket address'),
      '#description' => $this->t('Enter the full S3 bucket address including s3:// and the bucket name. Example: s3://my-bucket-name-identifier'),
      '#type' => 'textfield',
      '#size' => 80,
      '#maxlength' => 2000,
      '#default_value' => $this->config('dkan_dataset_archiver.settings')->get('remote_address') ?? '',
      '#states' => [
        'required' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory()->getEditable('dkan_dataset_archiver.settings');
    $config->set('archive', $form_state->getValue('archive'))
      ->set('archive_by_theme', $form_state->getValue('archive_by_theme'))
      ->set('archive_by_keyword', $form_state->getValue('archive_by_keyword'))
      ->set('create_annual_archives', $form_state->getValue('create_annual_archives'))
      ->set('include_manifest', $form_state->getValue('include_manifest'))
      ->set('archive_private', $form_state->getValue('archive_private'))
      ->set('storage_locations', $form_state->getValue('storage_locations'))
      ->set('local_years_retained', $form_state->getValue('local_years_retained'))
      ->set('remote_years_retained', $form_state->getValue('remote_years_retained'))
      ->set('remote_address', rtrim($form_state->getValue('remote_address'), '/ '))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
