<?php

namespace Drupal\dkan_dataset_archiver\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\dkan_dataset_archiver\CachedJsonResponseTrait;
use Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface;
use Drupal\dkan_dataset_archiver\Service\Util;
use League\Flysystem\Adapter\Local;
use League\Flysystem\Filesystem;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Archive API controller.
 */
class ArchiveApiController implements ContainerInjectionInterface {

  use CachedJsonResponseTrait;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The filesystem object.
   *
   * @var \League\Flysystem\Filesystem
   */
  private $filesystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManager
   */
  private $streamWrapperManager;

  /**
   * Cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(CacheBackendInterface $cacheBackend, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
    $this->cacheBackend = $cacheBackend;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('cache.default'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Get structured content of a directory.
   *
   * @param string $directory
   *   The directory to get content from.
   *
   * @return array
   *   The structured content.
   */
  private function getStructuredContentOfDirectory(string $directory): array {
    $content = $this->getFilesystem()->listContents($directory, TRUE);
    return $this->structuredContent($content);
  }

  /**
   * Structure content.
   *
   * @param array $content
   *   Array of files and folders.
   *
   * @return array
   *   Content structured by topic, then year.
   */
  private function structuredContent(array $content): array {
    $flatStructure = array_values(array_filter(array_map(function ($item) {
      return $this->setInfo($item);
    }, array_filter($content, function ($item) {
      return $item['type'] == 'file';
    })), function ($item) {
      $yearSet = isset($item['year']);
      $yearAllowed = $yearSet && $this->yearAllowed($item['year']);
      $topicSet = isset($item['topic']);
      return $yearSet && $yearAllowed && $topicSet;
    }));

    $structure = [];
    foreach ($flatStructure as $file) {
      $topic = $file['topic'];
      unset($file['topic']);
      $year = $file['year'];
      unset($file['year']);

      if (!isset($structure[$topic])) {
        $structure[$topic] = [];
      }

      if (!isset($structure[$topic][$year])) {
        $structure[$topic][$year] = [];
      }

      $structure[$topic][$year][] = $file;
    }

    return $structure;
  }

  /**
   * Restrict the number of years of archive returned.
   */
  private function yearAllowed(string $year): bool {
    $previous_year = (int) Util::date('-1 year')->format('Y');
    $years_to_keep = $this->archiverSettings->get('archive_years_retained');
    $oldest_year = $previous_year - $years_to_keep;
    return ((int) $year) > $oldest_year;
  }

  /**
   * Set item info.
   *
   * @param array $item
   *   Info on one file or folder.
   *
   * @return array
   *   Modified info.
   */
  private function setInfo(array $item): array {
    $info = [];
    $filePath = "public://{$item['path']}";
    $info['url'] = $this->fileCreateUrl($filePath);
    $info['size'] = $item['size'];
    [$month, $year] = $this->getMonthAndYearFromFilename($item['filename']);
    $info['type'] = isset($month) ? 'Monthly' : 'Annual';

    if (isset($month)) {
      $info['month'] = $month;
      $info['day'] = date('d', $item['timestamp']);
    }

    $info['year'] = $year;

    $info['topic'] = $this->getTopicFromDirectoryName($item['dirname']);

    return $info;
  }

  /**
   * Set the filesystem object.
   *
   * @param \League\Flysystem\Filesystem $filesystem
   *   The filesystem object.
   */
  public function setFileSystem(Filesystem $filesystem): void {
    $this->filesystem = $filesystem;
  }

  /**
   * Extract the month MM (if any) and year YYYY from end of filename.
   *
   * @param string $filename
   *   The filename.
   *
   * @return array
   *   Array with month (or NULL) and year (or NULL).
   */
  private function getMonthAndYearFromFilename(string $filename): array {
    if (preg_match('/_(\d\d)_(\d{4})$/', $filename, $matches)) {
      return [$matches[1], $matches[2]];
    }
    if (preg_match('/_(\d{4})$/', $filename, $matches)) {
      return [NULL, $matches[1]];
    }
    return [NULL, NULL];
  }

  /**
   * Extract the topic from the directory name.
   *
   * @param string $directoryName
   *   The directory name.
   *
   * @return string|null
   *   The topic, or NULL if it cannot be determined.
   */
  private function getTopicFromDirectoryName(string $directoryName): ?string {
    $pieces = explode('/', $directoryName);
    return $pieces[1] ?? NULL;
  }

  /**
   * Create a URL for a file URI.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return string
   *   The file URL on the public side of the site.
   */
  private function fileCreateUrl($uri): string {
    if ($wrapper = $this->streamWrapperManager->getViaUri($uri)) {
      $url = $wrapper->getExternalUrl();

      // @todo We need a solution to do this for all environments to convert
      // private edit domain urls to public.
      // Used to be dkan_dataset_archiver_force_url_to_public().
      // @todo Or maybe we don't need this anymore.  Contemplate removing.
      return $url;
    }
    return $uri;
  }

  /**
   * Get (or create) the filesystem object.
   *
   * @return \League\Flysystem\Filesystem
   *   The filesystem object.
   */
  private function getFilesystem(): Filesystem {
    if (!empty($this->filesystem)) {
      $this->filesystem = new Filesystem(new Local(Util::getDrupalPublicFilesDirectory()));
    }
    return $this->filesystem;
  }

  /**
   * Return info about a topic's most current zip file.
   *
   * @param string $topic
   *   Human-readable topic.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about a topic's current zip.
   */
  public function topicCurrentZip(string $topic): Response {
    $cacheTag = "topic:{$topic}:zip:current";
    // Check for that topic's current zip in existing cache.
    if ($cachedTopicCurrentZip = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedTopicCurrentZip);
    }
    $latest = $this->latestZip('archive/' . $topic);

    return $this->setCacheGetResponse($cacheTag, $latest);
  }

  /**
   * Return info about every topic's most current zip file.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about every topic's current zip.
   */
  public function allCurrentZips(): Response {
    // Check for that topic's current zip in existing cache.
    if ($cachedTopicCurrentZip = $this->getCacheTag('topics_current_zips')) {
      return $this->getCachedResponse($cachedTopicCurrentZip);
    }
    $latest = $this->latestZip('archive');

    return $this->setCacheGetResponse('topics_current_zips', $latest);
  }

  /**
   * Return info about the most current zip file in a directory.
   *
   * @param string $directory
   *   Directory to check.
   *
   * @return array
   *   Array of info about the most current zip file in the directory.
   */
  private function latestZip(string $directory): array {
    $contents = $this->getFilesystem()->listContents($directory, TRUE);
    $structure = [];

    $items = array_filter($contents, function ($item) {
      $dirFragments = explode('/', $item['dirname']);
      return 'current' === end($dirFragments);
    });

    $date = Util::date();
    $year = $date->format('Y');
    $month = $date->format('m');

    foreach ($items as $item) {
      $dirFragments = explode('/', $item['dirname']);
      if (isset($dirFragments[1])) {
        $structure[$dirFragments[1]][$year] = [
          'url' => $this->fileCreateUrl("public://{$item['path']}"),
          'size' => $item['size'],
          'type' => 'Monthly',
          'month' => $month,
        ];
      }
    }

    return $structure;
  }

  /**
   * Aggregate archives. For endpoint /api/1/archive/aggregate/.
   *
   * @param string $type
   *   The type of aggregation, either 'theme', keyword, 'annual_keyword' or
   *   'annual_theme'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   * @param string $link_type
   *   Set 'absolute' or 'relative' links, any other treated as 'absolute'.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about the archives.
   */
  public function aggregateArchives(string $type, string $filter = 'all', string $link_type = 'absolute'): Response {
    if (!$this->canShowArchivesBasedOnSettings($type)) {
      // The aggregation type and settings are not compatible.
      $msg = t('This type of aggregation is not allowed by settings.');
      return new Response($msg, 404);
    }
    $absolute = ($link_type === 'relative') ? FALSE : TRUE;
    // Check for that theme's archives in existing cache.
    $cacheTag = Util::getAggregationTag($type, $filter, $absolute);
    if (!empty($cacheTag) && $cachedThemeArchives = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedThemeArchives);
    }
    $ids = $this->aggregationQuery($type, $filter);

    $archives = $this->entityTypeManager->getStorage('dda_archive')->loadMultiple($ids);
    $structure = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    foreach ($archives as $archive) {
      $item = $this->assemblePayloadItem($archive, $type, $absolute);
      $structure[] = $item;
    }

    $payload = [
      'data' => $structure,
      'meta' => [
        'total_items' => $totalItems = count($structure),
        // @todo connect this to pagination.
        'current_page' => 1,
        'page_size' => -1,
        'total_pages' => (empty($totalItems)) ? 1 : ceil($totalItems / $totalItems),
      ],
    ];
    // Just in case we end up with no tag, make sure we end up with one.
    $cacheTag = (!empty($cacheTag)) ? $cacheTag : 'archive:never-to-be-read';
    // Aggregate archives don't change often so cache for a day (86400s).
    return $this->setCacheGetResponse($cacheTag, $payload, []);
  }

  /**
   * Individual archives. For endpoint /api/1/archive/individual/.
   *
   * @param string $link_type
   *   Set 'absolute' or 'relative' links, any other treated as 'absolute'.
   * @param string $filter_by
   *   Filter by either "keyword", "theme", or "none".
   * @param string $filter
   *   The actual filter value to look for in the filter_type.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about the archives.
   */
  public function individualArchives(string $link_type = 'absolute', string $filter_by = 'none', string $filter = 'all'): Response {
    if (!$this->canShowArchivesBasedOnSettings('individual')) {
      // The Archiving is not allowed.
      $msg = t('Archives are not allowed by settings.');
      return new Response($msg, 404);
    }
    $absolute = ($link_type === 'relative') ? FALSE : TRUE;
    // Check for archives in existing cache.
    $cacheTag = Util::getAggregationTag("individual", "{$filter_by}:{$filter}", $absolute);
    if (!empty($cacheTag) && $cachedThemeArchives = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedThemeArchives);
    }
    $ids = $this->individualQuery($filter_by, $filter);
    $archives = $this->entityTypeManager->getStorage('dda_archive')->loadMultiple($ids);
    $structure = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    foreach ($archives as $archive) {
      $item = $this->assemblePayloadItem($archive, 'individual', $absolute);
      $structure[] = $item;
    }

    $payload = [
      'data' => $structure,
      'meta' => [
        'total_items' => $totalItems = count($structure),
        // @todo connect this to pagination.
        'current_page' => 1,
        'page_size' => -1,
        'total_pages' => (empty($totalItems)) ? 1 : ceil($totalItems / $totalItems),
      ],
    ];
    // Just in case we end up with no tag, make sure we end up with one.
    $cacheTag = (!empty($cacheTag)) ? $cacheTag : 'archive:never-to-be-read';
    // Aggregate archives don't change often so cache for a day (86400s).
    return $this->setCacheGetResponse($cacheTag, $payload, []);
  }

  /**
   * Get archive IDs for an aggregation query.
   *
   * @param string $type
   *   The type of aggregation.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   *
   * @return array
   *   Array of dda_archive entity IDs.
   */
  protected function aggregationQuery(string $type, string $filter) : array {
    return ($type === 'annual') ? $this->annualAggregationQuery($filter) : $this->standardAggregationQuery($type, $filter);
  }

  /**
   * Get archive IDs for annual aggregations.
   *
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   *
   * @return array
   *   Array of dda_archive entity IDs.
   */
  protected function annualAggregationQuery(string $filter): array {
    // Truth table.
    // Filter   | Includes annual | Includes annual_subgroups
    // hospital |  no             | yes
    // all      | yes             | yes
    // none     | yes             | no.
    $query = $this->getBaseDdaArchiveQuery();
    $annual_subgroups = [];
    if (!in_array($filter, ['all', 'none'])) {
      // This is an odd edge case where annuals are being filtered by topic,
      // so it should not include the annual archives which are filter-less.
      $archive_types_include = [];
    }
    else {
      // Totally ok to include plain annuals in the search.
      $archive_types_include = ['annual'];
    }
    // None should not include any subgroups so it is pure 'annual'.
    if ($filter !== 'none') {
      $annual_subgroups = ['annual_keyword', 'annual_theme'];
    }
    $archive_types_include = array_merge($archive_types_include, $annual_subgroups);
    $query->condition('archive_type', $archive_types_include, 'IN');

    $query->sort('dataset_modified', 'DESC');

    if (($filter !== 'all') && ($filter !== 'none')) {
      $query->condition('aggregate_on', $filter);
      // Sub-sort by aggregation.
      $query->sort('aggregate_on', 'ASC');
    }

    $ids = $query->execute();
    return $ids;
  }

  /**
   * Get archive IDs for standard aggregations.
   *
   * @param string $type
   *   The type of aggregation (keyword, annual_keyword, theme, annual_theme).
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all'.
   *
   * @return array
   *   Array of archive node IDs.
   */
  protected function standardAggregationQuery(string $type, string $filter) : array {
    $query = $this->getBaseDdaArchiveQuery();
    $archive_types_include = [$type];
    // If type keyword or theme we should include the annuals for that type.
    $annual_subgroups = (in_array($type, ['keyword', 'theme'])) ? ["annual_{$type}"] : [];
    if ($filter === 'none') {
      // None should not include any subgroups, so it is pure type.
      $annual_subgroups = [];
    }
    $archive_types_include = array_merge($archive_types_include, $annual_subgroups);
    $query->condition('archive_type', $archive_types_include, 'IN');

    if ($filter !== 'all' && $filter !== 'none') {
      // These are special cases, do not sub-filter.
      $query->condition('aggregate_on', $filter);
    }
    $query->sort('aggregate_on', 'ASC')
      ->sort('dataset_modified', 'DESC');
    $ids = $query->execute();
    return $ids;
  }

  /**
   * Get archive IDs for standard aggregations.
   *
   * @param string $filter_by
   *   The field to filter by ('theme', 'keyword', 'none').
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all'.
   *
   * @return array
   *   Array of archive node IDs.
   */
  protected function individualQuery(string $filter_by, string $filter) : array {
    $query = $this->getBaseDdaArchiveQuery();
    $query->condition('archive_type', 'individual');

    if ($filter_by !== 'none' && $filter !== 'all') {
      // We need to filter by something.
      $query->condition("{$filter_by}s", $filter, '=');
    }

    $query->sort('dataset_modified', 'DESC')
      ->sort('name', 'ASC');
    $ids = $query->execute();
    return $ids;
  }

  /**
   * Check if aggregation archiving is enabled for the type of aggregation.
   *
   * @param string $type
   *   Type of aggregation, 'annual', 'theme', keyword, 'annual_keyword' or
   *   'annual_theme'.
   *
   * @return bool
   *   TRUE if theme archiving is disabled.
   */
  protected function canShowArchivesBasedOnSettings(string $type): bool {
    // First case to be TRUE calls it off.
    switch (TRUE) {
      case $this->archiverSettings->get('archive') !== '1':
      case $type === 'keyword' && $this->archiverSettings->get('archive_by_keyword') !== TRUE:
      case $type === 'theme' && $this->archiverSettings->get('archive_by_theme') !== TRUE:
      case ($type === 'annual_keyword') && (($this->archiverSettings->get('archive_by_keyword') !== TRUE) || $this->archiverSettings->get('create_annual_archives') !== TRUE):
      case ($type === 'annual_theme') && (($this->archiverSettings->get('archive_by_theme') !== TRUE) || $this->archiverSettings->get('create_annual_archives') !== TRUE):
      case $type === 'annual' && $this->archiverSettings->get('create_annual_archives') !== TRUE:
        return FALSE;

      default:
        // It made it this far, so it's OK.
        return TRUE;
    }
  }

  /**
   * Get a base DdaArchive query with common conditions.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   The base query.
   */
  protected function getBaseDdaArchiveQuery(): QueryInterface {
    /** @var \Drupal\Core\Entity\Query\QueryInterface $query */
    $query = $this->entityTypeManager->getStorage('dda_archive')->getQuery();
    $query->condition('status', 1);
    $query->accessCheck(TRUE);
    return $query;
  }

  /**
   * Assemble one item for the payload.
   *
   * @param \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive
   *   The archive entity.
   * @param string $type
   *   The type of aggregation.
   * @param bool $absolute
   *   Whether to make the URL absolute.
   *
   * @return array
   *   The payload item.
   */
  protected function assemblePayloadItem(DdaArchiveInterface $archive, string $type, bool $absolute): array {
    $item = [];
    // Local archive beats remote url if both exist.
    $local_archive = $archive->getResourceFileItems('local_archive');
    if (!empty($local_archive)) {
      $local_url = $local_archive[0]?->createFileUrl(!$absolute);
    }

    $remote_url_field_item = $archive->get('remote_url')->first();
    if ($remote_url_field_item) {
      /** @var \Drupal\link\LinkItemInterface $remote_url */
      // @phpstan-ignore-next-line
      $remote_url = $remote_url_field_item->uri;
    }

    $item['name'] = $archive->getName();
    $item['id'] = $archive->id();
    $item['type'] = $archive->getArchiveType();

    switch ($type) {
      case 'annual':
        $item['url'] = empty($local_url) ? $remote_url ?? '' : $local_url;

        break;

      case 'individual':
        $item['keywords'] = Util::getListOfMultiValues($archive->get('keywords'));
        $item['themes'] = Util::getListOfMultiValues($archive->get('themes'));
        $resources = Util::getListOfMultiValueUris($archive->get('resource_files'), $absolute);
        $item['resource_files'] = $resources;

        break;

      case 'theme':
      case 'annual_theme':
        $item['theme'] = $archive->get('aggregate_on')->value;
        $item['url'] = empty($local_url) ? $remote_url ?? '' : $local_url;
        break;

      case 'keyword':
      case 'annual_keyword':
        $item['keyword'] = $archive->get('aggregate_on')->value;
        $item['url'] = empty($local_url) ? $remote_url ?? '' : $local_url;
        break;

    }
    $item['size'] = $archive->get('size')->value;
    $item['date'] = $archive->get('dataset_modified')->value;
    $item['access_level'] = $archive->get('access_level')->value;

    return $item;
  }

}
