<?php

namespace Drupal\dkan_dataset_archiver\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Alt Text Validation Setting form.
 */
class ArchiverSettings extends ConfigFormBase {

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManager
   */
  private $streamWrapperManager;

  /**
   * Constructs a ArchiverSettings form.
   *
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManager $streamWrapperManager
   *   The stream wrapper manager.
   */
  public function __construct(StreamWrapperManager $streamWrapperManager) {
    $this->streamWrapperManager = $streamWrapperManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('stream_wrapper_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'dkan_dataset_archiver_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['dkan_dataset_archiver.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('dkan_dataset_archiver.settings');
    $form['#title'] = $this->t('DKAN Dataset Archiver Settings');

    $form['archive'] = [
      '#type' => 'radios',
      '#title' => $this->t('Archive datasets when updated, deleted, or unpublished.'),
      '#default_value' => $config->get('archive') ?? 0,
      '#options' => [
        '1' => $this->t('On'),
        '0' => $this->t('Off'),
      ],
    ];

    $form['datasets_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude specific datasets.'),
      '#description' => $this->t('Enter dataset IDs to exclude from archiving (one per line).'),
      '#default_value' => $config->get('datasets_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    $form['collation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('How to group archived datasets'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['collation']['archive_by_theme'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset theme'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Theme when updated, deleted, or unpublished during the same "Aggregation delay" window.'),
      '#default_value' => $config->get('archive_by_theme') ?? FALSE,
    ];

    $form['collation']['themes_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude by theme.'),
      '#description' => $this->t('Enter themes to exclude from archiving (one per line).'),
      '#default_value' => $config->get('themes_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['archive_by_keyword'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset keyword'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Keywords when updated, deleted, or unpublished during the same "Aggregation delay" window.'),
      '#default_value' => $config->get('archive_by_keyword') ?? FALSE,
    ];

    $form['collation']['keywords_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude by keyword.'),
      '#description' => $this->t('Enter keywords to exclude from archiving (one per line).'),
      '#default_value' => $config->get('keywords_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['create_annual_archives'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create annual archives of all public datasets.'),
      '#description' => $this->t('At the end of each year, a zip file will generate of all datasets for that year. If you’ve also selected “Archive by theme” or “Archive by keyword,” separate annual archives will be created for those as well.'),
      '#default_value' => $config->get('create_annual_archives') ?? FALSE,
    ];

    $form['collation']['create_current_download'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create a current download(s) of all aggregated datasets.'),
      '#description' => $this->t('Creates a zip file that continually updates with the latest aggregated datasets.'),
      '#default_value' => $config->get('create_current_download') ?? FALSE,
    ];

    $form['privacy'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Privacy'),
      '#weight' => 30,
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    $form['privacy']['archive_private'] = [
      '#type' => 'radios',
      '#title' => $this->t('Archive private datasets.'),
      '#default_value' => $config->get('archive_private') ?? 0,
      '#options' => [
        '0' => $this->t('Do not archive private datasets'),
        'in_public' => $this->t('Archive private datasets separately but still in the public file storage.'),
        'in_private' => $this->t('Archive private datasets separately in a private file storage. Requires Drupal private file storage be configured'),
      ],
      '#description' => $this->t('Creates separate archives for private datasets in addition to the public ones the users has chosen to generate. If you’ve also selected “Archive by theme” or “Archive by keyword,” private datasets will be grouped and archived the same way as public datasets.'),
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    // If private files are not configured, disable that option.
    if (!$this->isPrivateFilesStorageConfigured()) {
      $form['privacy']['archive_private']['in_private']['#disabled'] = TRUE;
    }

    $form['privacy']['treat_as_private'] = [
      '#type' => 'radios',
      '#title' => $this->t('Treat the following dataset accessLevels as private.'),
      '#default_value' => $config->get('treat_as_private') ?? 0,
      '#options' => [
        '0' => $this->t('Do not treat any AccessLevels as private. All archives will be public.'),
        'non-public' => $this->t("Only archives with accessLevel 'non-public' are treated as private."),
        'restricted public' => $this->t("Treat 'restricted public' and 'non-public' archives as private."),
        'public' => $this->t('Treat archives of all accessLevels as private.'),
      ],
      '#description' => $this->t('Sets the category of accessLevel to be treated as private determining whether the archive is stored in the private file system if it is available.'),
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    $form['collation']['aggregation_delay'] = [
      '#title' => $this->t('Aggregation delay'),
      '#description' => $this->t('The number of minutes to wait after a dataset is updated before creating an aggregated archive. If datasets bulk update automatically, this can be 10 minutes, if datasets update by hand, this should be a larger number to prevent premature aggregation. Minimum is 10 minutes, maximum is 1440 minutes (24 hours).'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 4,
      '#max' => 1440,
      '#min' => 10,
      '#default_value' => $config->get('aggregation_delay') ?? 10,
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['theme_keyword_map'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Map keywords or themes to other words.'),
      '#description' => $this->t("Enter keywords or themes that should be changed 'Original name -> New name' (one pair per line)."),
      '#default_value' => $config->get('theme_keyword_map'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['locations_group'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Archive Storage and Retention'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['storage_locations'] = [
      '#type' => 'radios',
      '#title' => $this->t('Storage location'),
      '#description' => $this->t('Choose where you would like to store the dataset archives.'),
      '#default_value' => $config->get('storage_locations') ?? 0,
      '#options' => [
        'local' => $this->t('Create only local archives %example', ['%example' => '(files/dkan_dataset_archives/)']),
        'remote' => $this->t('Create only remote archives'),
        'local_and_remote' => $this->t('Create archives in both local and remote locations'),
      ],
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['archive_years_retained'] = [
      '#prefix' => '<div class="container-inline">',
      '#title' => $this->t('Number of years to retain archives'),
      '#description' => $this->t('Set to 0 to retain indefinitely.'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 3,
      '#max' => 365,
      '#min' => 0,
      '#default_value' => $config->get('archive_years_retained') ?? 0,
      '#suffix' => '</div>',
    ];
    $form['locations_group']['remote_type'] = [
      '#title' => $this->t('Remote storage type'),
      '#description' => $this->t('Select the type of remote storage to use. Currently only supports AWS.'),
      '#type' => 'select',
      '#options' => [
        'aws:s3' => $this->t('AWS S3'),
      ],
      '#states' => [
        'required' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
      '#default_value' => $config->get('remote_type') ?? 'aws:s3',
    ];

    $form['locations_group']['remote_address'] = [
      '#title' => $this->t('The bucket address'),
      '#description' => $this->t('Enter the full bucket address including scheme and the bucket name. Example: s3://my-bucket-name-identifier'),
      '#type' => 'textfield',
      '#size' => 80,
      '#maxlength' => 2000,
      '#default_value' => $config->get('remote_address') ?? '',
      '#states' => [
        'required' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];
    $form['locations_group']['remote_region'] = [
      '#title' => $this->t('Region'),
      '#description' => $this->t('Enter the AWS region for the bucket. Example: us-east-1'),
      '#type' => 'textfield',
      '#size' => 80,
      '#maxlength' => 100,
      '#default_value' => $config->get('remote_region') ?? 'us-east-1',
      '#states' => [
        'required' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory()->getEditable('dkan_dataset_archiver.settings');
    $config->set('archive', $form_state->getValue('archive'))
      ->set('archive_by_theme', $form_state->getValue('archive_by_theme'))
      ->set('archive_by_keyword', $form_state->getValue('archive_by_keyword'))
      ->set('create_annual_archives', $form_state->getValue('create_annual_archives'))
      ->set('create_current_download', $form_state->getValue('create_current_download'))
      ->set('archive_private', $form_state->getValue('archive_private'))
      ->set('treat_as_private', $form_state->getValue('treat_as_private'))
      ->set('aggregation_delay', $form_state->getValue('aggregation_delay'))
      ->set('storage_locations', $form_state->getValue('storage_locations'))
      ->set('archive_years_retained', $form_state->getValue('archive_years_retained'))
      ->set('remote_type', $form_state->getValue('remote_type'))
      ->set('remote_address', rtrim($form_state->getValue('remote_address'), '/ '))
      ->set('remote_region', $form_state->getValue('remote_region'))
      ->set('themes_to_skip', $this->convertSort($form_state->getValue('themes_to_skip')))
      ->set('keywords_to_skip', $this->convertSort($form_state->getValue('keywords_to_skip')))
      ->set('datasets_to_skip', $this->convertSort($form_state->getValue('datasets_to_skip')))
      ->set('theme_keyword_map', $this->convertSortArrowFormat($form_state->getValue('theme_keyword_map')))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Dedupe, sort and convert to one per line.
   *
   * @param string $multi_values_raw
   *   A comma or new line separated string of multiple values.
   *
   * @return string
   *   A new-line separated string that has been deduped and sorted.
   */
  private function convertSort($multi_values_raw): string {
    // Clean up any bad data by converting all separators into ','.
    $multi_values = str_replace(["\r\n", "\n", "\r", ', ', ' ,'], ',', $multi_values_raw);
    $values = explode(',', $multi_values);
    // Dedupe and sort.
    $values = array_unique($values);
    natcasesort($values);
    $multi_values_sorted = implode("\n", $values);
    return $multi_values_sorted;
  }

  /**
   * Dedupe, sort and convert while sanitizing the arrows.
   *
   * @param string $multi_values_raw
   *   A comma or new line separated string of multiple values.
   *
   * @return string
   *   A new-line separated string that has been deduped and sorted.
   */
  private function convertSortArrowFormat($multi_values_raw): string {
    // Clean up any bad data by converting all separators into ','.
    $multi_values = str_replace(["\r\n", "\n", "\r", ', ', ' ,'], ',', $multi_values_raw);
    $values = explode(',', $multi_values);
    // Dedupe and clean.
    foreach ($values as &$value) {
      $parts = explode('->', $value);
      if (count($parts) === 2) {
        $value = trim($parts[0]) . ' -> ' . trim($parts[1]);
        if (empty(trim($parts[0])) || empty(trim($parts[1]))) {
          // One side is missing so remove it.
          $value = '';
        }
      }
      else {
        // If it doesn't have exactly one arrow, remove it to avoid problems.
        $value = '';
      }
    }
    // Must unset the reference.
    unset($value);
    $values = array_unique($values);
    natcasesort($values);
    $multi_values_sorted = implode("\n", $values);
    return $multi_values_sorted;
  }

  /**
   * Determine if private file storage is configured.
   *
   * @return bool
   *   TRUE if private file storage is configured, FALSE otherwise.
   */
  protected function isPrivateFilesStorageConfigured(): bool {
    return $this->streamWrapperManager->isValidUri('private://');
  }

}
