<?php

namespace Drupal\dkan_dataset_archiver\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\DelayedRequeueException;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Queue\RequeueException;
use Drupal\Core\Queue\SuspendQueueException;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines 'archive_remote_file_sync' queue worker.
 */
#[QueueWorker(
  id: 'archive_remote_file_sync',
  title: new TranslatableMarkup('Sync local files to S3.'),
  cron: ['time' => 30]
)]
class SyncToRemote extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * Constructs a SyncToRemote object.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dkan_dataset_archiver\Service\ArchiveService $archive_service
   *   The archive service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ArchiveService $archiveService) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->archiveService = $archiveService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dkan_dataset_archiver.archive_service')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function processItem($data): void {
    try {
      $this->archiveService->syncToRemote($data);
    }
    catch (SuspendQueueException $e) {
      // Re-throw this so it bubbles up.
      throw $e;
    }
    catch (\Exception $e) {
      // Requeue the item for later processing.
      throw new RequeueException('Failed to sync to remote. Requeuing item.', 0, $e);
    }

  }

}
