<?php

namespace Drupal\dkan_dataset_archiver\Service;

use Drupal\common\DatasetInfo;
use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Queue\DelayedRequeueException;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\SuspendQueueException;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\dkan_dataset_archiver\AwsS3Trait;
use Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface;
use Drupal\dkan_dataset_archiver\HelperTrait;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\metastore_search\Search;
use Drupal\metastore\Storage\DataFactory;
use Drupal\metastore\Storage\NodeData;
use Drupal\node\NodeInterface;
use Procrastinator\Result;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Finder\Finder;


use function PHPUnit\Framework\throwException;

/**
 * Archive Service.
 */
class ArchiveService implements ContainerInjectionInterface {

  use AwsS3Trait;
  use HelperTrait;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * Dkan datasetInfo.
   *
   * @var \Drupal\common\DatasetInfo
   */
  private $datasetInfo;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The metastore search api service.
   *
   * @var \Drupal\metastore_search\Search
   */
  private $metastoreSearch;

  /**
   * Metastore Storage factory.
   *
   * @var \Drupal\metastore\Storage\DataFactory
   */
  protected $metastoreStorage;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;


  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * Storages.
   *
   * @var array
   */
  protected $storages = [];

  /**
   * Term map to convert or coalesce terms.
   *
   * @var array
   */
  protected $termMap = [];

  /**
   * Utility of helper functions.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\Util
   */
  protected $util;

  /**
   * Current year.
   *
   * @var string
   */
  private $year;

  /**
   * All published datasets.
   *
   * @var array
   */
  protected $publishedDatasets = [];

  /**
   * {@inheritDoc}
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\dkan_dataset_archiver\Service\Util $util
   *   Utility.
   * @param \Drupal\common\DatasetInfo $datasetInfo
   *   DKAN datasetInfo.
   * @param \Drupal\metastore_search\Search $metastoreSearchService
   *   Metastore Search wrapper for the SearchApi.
   * @param \Drupal\metastore\Storage\DataFactory $metastoreStorage
   *   Metastore Storage factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver logger channel.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Queue\QueueFactory $queue
   *   The queue factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Util $util,
    DatasetInfo $datasetInfo,
    Search $metastoreSearchService,
    DataFactory $metastoreStorage,
    EntityTypeManagerInterface $entityTypeManager,
    FileRepositoryInterface $file_repository,
    FileSystemInterface $file_system,
    LoggerInterface $logger,
    ModuleHandler $moduleHandler,
    QueueFactory $queue,
  ) {
    $this->datasetInfo = $datasetInfo;
    $this->queue = $queue;
    $this->moduleHandler = $moduleHandler;
    $this->util = $util;
    $this->year = Util::date()->format('Y');
    $this->logger = $logger;
    $this->metastoreSearch = $metastoreSearchService;
    $this->entityTypeManager = $entityTypeManager;
    $this->fileRepository = $file_repository;
    $this->fileSystem = $file_system;
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
    $this->metastoreStorage = $metastoreStorage;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('dkan_dataset_archiver.util'),
      $container->get('dkan.dataset_info'),
      $container->get('dkan.metastore_search.service'),
      $container->get('dkan.metastore.storage'),
      $container->get('entity_type.manager'),
      $container->get('file.repository'),
      $container->get('file_system'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('module_handler'),
      $container->get('queue'),
    );
  }

  /**
   * Ensure archive year is allowed.
   *
   * @param string $directory
   *   Directory containing the archive.
   * @param int $oldest_year
   *   Oldest year allowed.
   *
   * @return bool
   *   TRUE if the year is within the allowed range.
   */
  private function archiveYearAllowed(string $directory, int $oldest_year): bool {
    if (empty($directory)) {
      return FALSE;
    }
    $pieces = explode('/', $directory);
    if (empty($pieces)) {
      return FALSE;
    }
    $year = end($pieces);
    return is_numeric($year) && (int) $year > $oldest_year;
  }

  /**
   * Queue aggregate archiving of a theme or keywords's individual archives.
   *
   * @param string $type
   *   Either 'keyword' or 'theme'.
   * @param string $term
   *   The term to aggregate on.
   * @param array $archive_ids
   *   The archive ids to include in the aggregation.
   * @param bool $private
   *   Whether the aggregation is private.
   * @param string|null $aggregate_of
   *   The collation type to use for the aggregation. Usually matches type.
   */
  public function addToAggregationQueue($type, $term, $archive_ids, $private = FALSE, ?string $aggregate_of = NULL): void {
    /** @var \Drupal\Core\Queue\QueueInterface $archiveQueue */
    $archiveQueue = $this->queue->get('archive_aggregation');
    $aggregation_data = [
      'type' => $type,
      'aggregate_of' => $aggregate_of,
      'term' => $term,
      'archive_ids' => $archive_ids,
      'private' => $private,
    ];
    $queue_id = $archiveQueue->createItem($aggregation_data);
    $private_text = $private ? t('private') . ' ' : '';

    $this->logger->info(
      "@type @term had queued archives '@ids' for aggregation queue_id: %queueId", [
        '@type' => "{$private_text}{$type}",
        '@term' => $term,
        '@ids' => implode(',', $archive_ids),
        '%queueId' => $queue_id,
      ]
    );
  }

  /**
   * Add an archive to the remote file sync queue.
   *
   * @param \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive
   *   The archive entity.
   */
  public function addToRemoteFileSyncQueue(DdaArchiveInterface $archive): void {
    if (in_array($this->archiverSettings->get('storage_locations'), ['remote', 'local_and_remote'])) {
      $data = [
        'archive_id' => $archive->id(),
        'title' => $archive->getName(),
      ];
      /** @var \Drupal\Core\Queue\QueueInterface $remoteFileSyncQueue */
      $remoteFileSyncQueue = $this->queue->get('archive_remote_file_sync');
      // @todo Consider checking to see if the file actually changed.
      $queue_id = $remoteFileSyncQueue->createItem($data);
      $this->logger->info(
        "Archive '@title' (ID: @id) queued for remote file sync, queue_id: %queueId", [
          '@title' => $archive->getName(),
          '@id' => $archive->id(),
          '%queueId' => $queue_id,
        ]
      );
    }
  }

  /**
   * Create an aggregate archive for a theme or keyword.
   *
   * This is called by the ArchiveAggregation queue worker.
   *
   * @param array $data
   *   The aggregation data containing:
   *   - type: keyword, theme, annual, annual_keyword, annual_theme.
   *   - term: the term to aggregate on.
   *   - archive_ids: the archive ids to include in the aggregation.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function createAggregateArchive(array $data): bool {
    if ($this->archiverSettings->get('archive') !== '1') {
      // Archiving is turned off, so bail out.
      return FALSE;
    }
    $type = $data['type'] ?? '';
    $aggregate_of = $data['aggregate_of'];
    $term = $data['term'] ?? '';
    $archive_ids = $data['archive_ids'] ?? [];
    $private = $data['private'] ?? FALSE;

    $msg_data = [
      '@type' => $type,
      '@term' => (!empty($term)) ? $term : t('unspecified'),
      '@private' => $private,
      '@ids' => implode(', ', $data['archive_ids']),
    ];
    if (!empty($data['type']) && !empty($data['archive_ids'])) {
      $this->logger->info('Creating aggregate archive for @type: @term with archives @ids', $msg_data);
      if ($type === 'current') {
        // Current needs to pull file data from node:data:datasets.
        $zip_info = $this->createAggregatedZipFile($type, $aggregate_of, $term, $this->aggregateArchiveFiles($archive_ids, 'dataset'), $private);
      }
      else {
        // Anything non 'current' needs to pull data from archives.
        $zip_info = $this->createAggregatedZipFile($type, $aggregate_of, $term, $this->aggregateArchiveFiles($archive_ids, 'dda_archive'), $private);
      }

      // Need to turn this into a file entity.
      $file = File::create([
        'uri' => $zip_info['drupal_file_uri'],
        'filename' => basename($zip_info['file_url']),
        'filemime' => 'application/zip',
        'status' => 1,
      ]);
      $file->save();
      // Create DdaArchive.
      if ($type === 'current') {
        // Currents always update an existing archive if one exists.
        $archive = $this->makeGetCurrentArchive($data, $term, $private, $zip_info, $file, $archive_ids);
      }
      else {
        $archive = $this->makeNewAggregateArchive($data, $term, $private, $zip_info, $file, $archive_ids);
      }
      $archive->setSyncing(TRUE);
      $archive->save();
      return TRUE;
    }
    else {
      $this->logger->error("Invalid data sent to createAggregateArchive(). type: '@type' term: '@term' with archives ids: '@ids'", $msg_data);
      // In the case of bad data, we do not want to keep this in the queue,
      // so return TRUE to indicate it can be removed from the queue.
      return TRUE;
    }
  }

  /**
   * Get or create a 'current' aggregate archive entity.
   *
   * @param array $data
   *   The aggregation data containing:
   *   - type: keyword, theme, annual, annual_keyword, annual_theme.
   *   - term: the term to aggregate on.
   *   - archive_ids: the archive ids to include in the aggregation.
   * @param string $term
   *   The term to aggregate on.
   * @param bool $private
   *   Whether the aggregation is private.
   * @param array $zip_info
   *   Information about the created zip file including:
   *   - file_url: the URL to the zip file.
   *   - drupal_file_uri: the Drupal file URI of the zip file.
   *   - file_size: the size of the zip file in bytes.
   * @param \Drupal\file\FileInterface $file
   *   The file entity for the zip file.
   * @param array $archive_ids
   *   An array of archive node IDs to include in the aggregation.
   *
   * @return \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface
   *   The created DdaArchive entity.
   */
  protected function makeGetCurrentArchive(array $data, string $term, bool $private, array $zip_info, FileInterface $file, array $archive_ids): DdaArchiveInterface {
    // We have to try to load a possibly existing 'current' and modify it
    // with a new revision.
    $storage = $this->entityTypeManager->getStorage('dda_archive');
    $properties = [
      'archive_type' => 'current',
      'aggregate_of' => $data['aggregate_of'],
      'aggregate_on' => $term,
    ];
    if ($private) {
      $properties['access_level'] = Util::getAccessLevelsThatAreConsideredPrivate();
    }
    else {
      $properties['access_level'] = Util::getAccessLevelsThatAreConsideredPublic();
    }
    $existing = $storage->loadByProperties($properties);
    if (!empty($existing)) {
      /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
      $archive = reset($existing);
      // Update the existing archive.
      $archive->set('dataset_modified', Util::date()->format('Y-m-d'));
      $archive->set('size', $zip_info['file_size'] ?? 0);
      $archive->set('source_archives', Util::buildEntityReferenceTargets($archive_ids));
      $archive->setNewRevision(TRUE);
      $archive->setRevisionCreationTime(Util::date()->getTimestamp());
      $archive->setRevisionLogMessage("Updated due to recent saves to datasets in the $term group.");
      // Update the file entity reference.
      $archive->set('resource_files', [
        'target_id' => $file->id(),
        'display' => 1,
      ]);
      return $archive;
    }
    else {
      // Create a new one.
      return $this->makeNewAggregateArchive($data, $term, $private, $zip_info, $file, $archive_ids);
    }
  }

  /**
   * Create a new aggregate archive entity.
   *
   * @param array $data
   *   The aggregation data containing:
   *   - type: keyword, theme, annual, annual_keyword, annual_theme.
   *   - term: the term to aggregate on.
   *   - archive_ids: the archive ids to include in the aggregation.
   *   - private: whether the aggregation is private.
   * @param string $term
   *   The term to aggregate on.
   * @param bool $private
   *   Whether the aggregation is private.
   * @param array $zip_info
   *   Information about the created zip file including:
   *   - file_url: the URL to the zip file.
   *   - drupal_file_uri: the Drupal file URI of the zip file.
   *   - file_size: the size of the zip file in bytes.
   * @param \Drupal\file\FileInterface $file
   *   The file entity for the zip file.
   * @param array $archive_ids
   *   An array of archive node IDs to include in the aggregation.
   *
   * @return \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface
   *   The created DdaArchive entity.
   */
  protected function makeNewAggregateArchive(array $data, string $term, bool $private, array $zip_info, FileInterface $file, array $archive_ids): DdaArchiveInterface {
    $archive_data = [
      'name' => "{$data['type']}: {$data['term']} " . Util::date()->format('Y-m-d H:i'),
      'archive_type' => $data['type'],
      'dataset_modified' => Util::date()->format('Y-m-d'),
      'aggregate_of' => $data['aggregate_of'],
      'aggregate_on' => $data['term'],
      'themes' => $data['type'] === 'theme' ? [$term] : [],
      'keywords' => $data['type'] === 'keyword' ? [$term] : [],
      // We only have a bool to work with, and private might be one of
      // multiple levels, so we set to either the highest private
      // 'non-public' or the lowest public.
      'access_level' => $private ? 'non-public' : 'public',
      'size' => $zip_info['file_size'] ?? 0 ,
      'status' => '1',
      'source_archives' => Util::buildEntityReferenceTargets($archive_ids),

      'resource_files' => [
        'target_id' => $file->id(),
        'display' => 1,
      ],
      // @todo should be the s3 url if remotes are allowed but maybe adding
      // it should wait until after the mover queue moves it.
      // 'remote_url' = ??
    ];

    $storage = $this->entityTypeManager->getStorage('dda_archive');
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    $archive = $storage->create($archive_data);
    return $archive;
  }

  /**
   * Aggregate files from a list of archive ids and build a manifest.
   *
   * @param array $archive_ids
   *   The DdaArchive node ids to aggregate files from.
   * @param string $source_type
   *   The source to use for aggregation (e.g. 'dataset', 'dda_archive').
   *
   * @return array
   *   Array containing the file URLs and manifest text ['files', 'manifest'].
   */
  protected function aggregateArchiveFiles(array $archive_ids, string $source_type): array {
    $files = [];
    $manifest = [];
    if (!empty($archive_ids)) {
      if ($source_type === 'dataset') {
        $storage = $this->entityTypeManager->getStorage('node');
        $sources = $storage->loadMultiple($archive_ids);
        /** @var \Drupal\node\NodeInterface $source */
      }
      else {
        $storage = $this->entityTypeManager->getStorage('dda_archive');
        $sources = $storage->loadMultiple($archive_ids);
        /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchive $source */
      }

      foreach ($sources as $source) {
        $manifest_item = [
          // Discovery pattern of archive source : dataset source.
          'name' => ($source instanceof DdaArchiveInterface) ? $source->getName() : $source->getTitle(),
          'dataset_id' => ($source instanceof DdaArchiveInterface) ? $source->get('dataset_id')->value : Util::grabMetadata($source, 'identifier'),
          'type' => ($source instanceof DdaArchiveInterface) ? $source->getArchiveType() : 'current',
          'modified_date' => ($source instanceof DdaArchiveInterface) ? $source->get('dataset_modified')->value : Util::grabMetadata($source, 'modified'),
          // @todo private and access level do not agree.
          'access_level' => ($source instanceof DdaArchiveInterface) ? $source->get('access_level')->value : Util::grabMetadata($source, 'accessLevel'),
          'private' => ($source instanceof DdaArchiveInterface) ? (bool) $source->isPrivate() : Util::isConsideredPrivate(Util::grabMetadata($source, 'accessLevel')),
          'resources' => [],
        ];
        if ($source_type === 'dataset') {
          $resource_files = Util::grabMetadata($source, 'distribution');
        }
        else {
          $resource_files = $source->getResourceFileItems();
        }

        if (!empty($resource_files)) {
          foreach ($resource_files as $resource_file) {
            if ($resource_file instanceof FileInterface) {
              $file_url = $resource_file->createFileUrl(FALSE);
              $files[] = $file_url;
              $entry = [
                'filename' => $resource_file->getFilename(),
                'filesize' => $resource_file->getSize(),
                'mime_type' => $resource_file->getMimeType(),
              ];
              $manifest_item['resources'][] = $entry;
            }
            else {
              // This is directly in a dataset.
              $file_url = $resource_file->downloadURL;
              $headers = get_headers($file_url, TRUE);
              if (isset($headers['Content-Length'])) {
                $fileSize = $headers['Content-Length'];
              }

              $files[] = $file_url;
              $entry = [
                'filename' => basename($file_url),
                'filesize' => $fileSize ?? NULL,
                'mime_type' => $resource_file->mediaType,
              ];
              $dictionary_url = $resource_file->describedBy ?? '';
              if ($dictionary_url) {
                $files[] = $dictionary_url;
                $entry['describedBy'] = basename($dictionary_url);
              }
              $manifest_item['resources'][] = $entry;
            }
          }
        }
        $manifest[] = $manifest_item;
      }
    }
    // Remove duplicates from files since dictionaries may overlap.
    $files = array_unique($files);
    $manifest_text = json_encode($manifest, JSON_PRETTY_PRINT);
    return ['files' => $files, 'manifest' => $manifest_text];
  }

  /**
   * Clears cache and purges archive endpoints. //@todo deprecated?
   *
   * @param string $type
   *   The type of the archive theme, keyword, annual_theme, annual_keyword.
   * @param string $theme_or_keyword
   *   The theme or keyword aggregating the archive.
   */
  public function clearCacheAndPurge(string $type, string $theme_or_keyword): void {
    // @todo fix all the routes and cache tags..
    switch ($type) {
      case 'keyword':
      case 'theme':
        $routes = [
          "api/1/archive/{$type}/{$theme_or_keyword}",
          "api/1/archive/{$type}/{$theme_or_keyword}/current-zip",
        ];
        $cache_tags = ["{$type}_archives", "{$type}_current_zips"];
        break;

      case 'annual_keyword':
        $routes = [
          "api/1/archive/keyword/{$theme_or_keyword}",
          "api/1/archive/keyword/{$theme_or_keyword}/annual",
        ];
        $cache_tags = ["keyword_archives", "keyword_current_zips"];
        break;

      case 'annual_theme':
        $routes = [
          "api/1/archive/theme/{$theme_or_keyword}",
          "api/1/archive/theme/{$theme_or_keyword}/annual",
        ];
        $cache_tags = ["theme_archives", "theme_current_zips"];
        break;

      case 'annual':
        $routes = [
          "api/1/archive/annual/",
        ];
        $cache_tags = ["annual_archives"];
        break;
    }
    // Clear relevant caches.
    // @todo rework this without purge service or optional.
    // ->archivePurgeService->clearUrls($archive_api_routes, $cache_tags);
  }

  /**
   * Create an individual zip for a given url. @todo consider deprecate.
   *
   * @param string $url
   *   The url to create a zip for.
   */
  public function createIndividualZip($url): void {
    if ($url) {
      $publicFiles = $this->util->getDrupalPublicFilesDir();
      $cleanUrl = explode("sites/default/files/", $url)[1];
      $fileSrc = "{$publicFiles}/{$cleanUrl}";
      $fileDest = $fileSrc . '.zip';
      $filename = explode('/', $cleanUrl);
      $filename = end($filename);

      $zip = new \ZipArchive();
      $zip->open($fileDest, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
      $zip->addFile($fileSrc, $filename);
      $zip->close();
      $this->logger->notice(
        "Zip for %fileDest created.", [
          '%fileDest' => $cleanUrl,
        ]
      );
    }
  }

  /**
   * Create an individual zip by uuid.
   *
   * @param string $uuid
   *   The uuid of the dataset to zip.
   */
  public function createIndividualZipByUuid(string $uuid): void {
    $datasetInfo = $this->datasetInfo->gather($uuid);
    if (!isset($datasetInfo['notice'])) {
      $revision = $datasetInfo['latest_revision'];
      if (isset($datasetInfo['published_revision'])) {
        $revision = $datasetInfo['published_revision'];
      }
      if (count($revision['distributions'])) {
        foreach ($revision['distributions'] as $distribution) {
          $this->createIndividualZip($distribution['file_path']);
        }
      }
    }
    else {
      $this->logger->notice("Uuid not found.");
    }
  }

  /**
   * Create a $theme_or_keyword's aggregated zip file.
   *
   * @param string $type
   *   Archive type 'annual', 'individual', or 'aggregate'.
   * @param string|null $aggregation_type
   *   The aggregation type 'keyword' or 'theme'.
   * @param string $theme_or_keyword
   *   Theme or keyword.
   * @param array $zip_contents
   *   Array containing the file URLs and manifest text ['files', 'manifest'].
   * @param bool $private
   *   Whether the archive is private or not.
   *
   * @return array
   *   Contains the following elements:
   *   ['file_url', 'drupal_file_uri', 'file_size', 'private'].
   */
  protected function createAggregatedZipFile(string $type, ?string $aggregation_type, string $theme_or_keyword, array $zip_contents, bool $private): array {
    $zip_info = $this->archivePathAndFilename($type, $aggregation_type, $theme_or_keyword, '', $private);
    $file_url = "{$zip_info['directory']}/{$zip_info['filename']}";
    $zip = new \ZipArchive();
    $worked = $zip->open("{$zip_info['directory']}/{$zip_info['filename']}", \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
    $this->addDownloadUrlFilesToZip($zip_contents, $zip, $private);
    $this->addOtherFilesToAggregateArchive($theme_or_keyword, $zip, $zip_contents['manifest'] ?? '');
    $success = $zip->close();
    if (!$success) {
      $this->logger->error(
        'Failed to create aggregated archive on %type: %theme - %file_url.', [
          '%type' => $aggregation_type,
          '%theme' => $theme_or_keyword,
          '%file_url' => $file_url,
        ]
      );
      return [];
    }
    if (file_exists($file_url)) {
      // Confirmed: We have a file created.
      $file_size = filesize($file_url);
      $this->logger->notice('Created aggregated archive on %type: %theme - %file_url.', [
        '%type' => $type,
        '%theme' => $theme_or_keyword,
        '%file_url' => $file_url,
      ]);

      return [
        'file_url' => $file_url,
        'drupal_file_uri' => "{$zip_info['drupal_directory']}/{$zip_info['filename']}",
        'file_size' => $file_size,
        'private' => $private,
      ];
    }
    else {
      // The zip files for some reason is not there.
      $this->logger->error('Failed to create aggregated archive on %type: %theme - %file_url.', [
        '%type' => $type,
        '%theme' => $theme_or_keyword,
        '%file_url' => $file_url,
      ]);
      return [];
    }
  }

  /**
   * Create a theme_or_keyword's download-all zip.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword. @todo Rework this to be about Theme or keyword.
   */
  public function createDownloadAll(string $theme_or_keyword): void {
    $publicFolder = $this->util->getDrupalPublicFilesDir();
    $zipPath = $publicFolder . "/archive/{$theme_or_keyword}/current";
    $this->util->prepareDir($zipPath);

    $machine_name = strtolower(str_replace(' ', '_', $theme_or_keyword));
    $zipFilename = "{$machine_name}_current_data.zip";

    $zip = new \ZipArchive();
    $zip->open("{$zipPath}/{$zipFilename}", \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
    // @todo work out the manifest
    $manifest = "I need to be worked out.";
    $downloadUrls = $this->getThemeDownloadUrls($theme_or_keyword);
    $zip_contents = [
      'files' => $downloadUrls,
      'manifest' => $manifest,
    ];
    // $this->addDownloadUrlFilesToZip($zip_contents, $zip);
    // $this->addOtherFilesToAggregateArchive($theme_or_keyword, $zip);
    $this->logger->notice(
      'Created download-all zip for %theme in %path.', [
        '%theme' => $theme_or_keyword,
        '%path' => $zipPath,
      ]
    );
  }

  /**
   * Add files from a list of urls to a zip archive.
   *
   * @param array $zip_contents
   *   Contains an array of files and a json manifest ['files', 'manifest'].
   * @param \ZipArchive $zip
   *   Zip archive.
   * @param bool $private
   *   Whether the archive is private or not.
   */
  protected function addDownloadUrlFilesToZip(array $zip_contents, \ZipArchive $zip, bool $private): void {
    $file_urls = $zip_contents['files'] ?? [];
    foreach ($file_urls as $file_url) {
      [$csvPath, $csvFilename] = $this->csvPathAndFilename($file_url, $private);
      if (isset($csvPath) && isset($csvFilename)) {
        $zip->addFile("{$csvPath}/{$csvFilename}", $csvFilename);
      }
    }
  }

  /**
   * Sync files to Remote.
   *
   * @param array $data
   *  Array with achive id.
   */
  public function syncToRemote(array $data): void {
    if (!$this->canSync()) {
      // Sync is not enabled, so bail out silently.
      return;
    }

    $remote_type = $this->archiverSettings->get('remote_type');
    $remote_address = $this->archiverSettings->get('remote_address');
    // Sort out what kind of remote.  Currently only supports AWS, but the
    // structure is here to expand.
    switch (TRUE) {
      case ($remote_type === 'aws:s3' && !empty($remote_address)):
        $fileSystem = $this->getAwsS3Filesystem();
        break;

      case (empty($remote_address)):
        $this->logger->error('Remote address not set for remote sync. Unable to sync.');
        throw new SuspendQueueException("Remote address not set for remote sync. Unable to sync.");
        return;

      default:
        $this->logger->error('Remote type @type not supported for remote sync.', ['@type' => $remote_type]);
        throw new \Exception("Remote type $remote_type not supported for remote sync.");
        return;

    }
    // Load the archive from data.
    $archive_id = $data['archive_id'] ?? NULL;
    $storage = $this->entityTypeManager->getStorage('dda_archive');
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    $archive = $storage->load($archive_id);
    // Get the files.
     $file_base = ($archive->isPrivate()) ? Util::getDrupalPrivateFilesDirectory() : Util::getDrupalPublicFilesDirectory();
     $resource_files = $archive->getResourceFileItems();
    // Grab and alter the paths.
    foreach ($resource_files as $resource_file) {
      // @todo not right.  should use the existing paths from the files.
      /** @var \Drupal\file\FileInterface $resource_file */
      $filePath = "{$resource_file->getFileUri()}";
      $destination = $resource_file->getFileUri();
      $destination = str_replace(['public://', 'private://'], "{$remote_address}/", $destination);
      // Adjust for private or public destination.
      $replacement = ($archive->isPrivate()) ? '/dataset_archives/private/' : '/dataset_archives/public/';
      $destination = str_replace(['/dataset-archives/private/', '/dataset-archives/'], $replacement, $destination);
  //@todo LEFT OFF HERE!!!!!


       // @todo this is not right.  Disconnected.
      $stream = fopen($destination, 'w');
      stream_set_chunk_size($stream, 10 * 1000 * 1000);
      $success = copy($filePath, $destination);
      if (!$success) {
        $this->logger->error(
          'Failed to copy file from %source to %destination for archive ID: @id.', [
            '%source' => $filePath,
            '%destination' => $destination,
            '@id' => $archive_id,
          ]
        );
      }
      else {
        $this->logger->notice(
          'Successfully copied file from %source to %destination for archive ID: @id.', [
            '%source' => $filePath,
            '%destination' => $destination,
            '@id' => $archive_id,
          ]
        );
      }

    // DO we ned this>?  fclose($stream);
      // VALIDATE THE COPY SUCCEEDED.

    // Copy them each to the remote.
    }

    // @todo Optionally remove local files if sync was successful.
    if ($this->archiverSettings->get('storage_locations') === 'remote') {
      // Remove local files from the archive .

      // Remove local files from the filesystem.

    }

    // @todo save the archive with the new locations added.
    // Add a meaningful revision message.  (somehow keep this save from triggering a a new sync)
  }

  /**
   * Determine if remote sync is possible.
   *
   * @return bool
   *   TRUE if remote sync can occur.
   */
  protected function canSync (): bool {
    $remote_sync = $this->archiverSettings->get('storage_locations');
    $remote_allowed = in_array($remote_sync, ['remote', 'local_and_remote']);
    return $remote_allowed;
  }

  /**
   * Add other files to the aggregate archive.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword.
   * @param \ZipArchive $zip
   *   Archive.
   * @param string $manifest
   *   Manifest text.
   */
  private function addOtherFilesToAggregateArchive(string $theme_or_keyword, \ZipArchive $zip, string $manifest = ''): void {
    if (!empty($manifest)) {
      // Add the manifest to the zip file.
      $zip->addFromString('manifest.json', $manifest);
    }
  }

  /**
   * Return an archive's path, drupal path, and filename based on archive type.
   *
   * @param string $type
   *   Archive type: 'keyword', 'annual_keyword', 'theme', 'annual_theme',
   *   'annual'.
   * @param string|null $aggregation_type
   *   The type of aggregation: 'keyword' or 'theme'.
   * @param string $theme_or_keyword
   *   Theme or keyword to aggregate on.
   * @param string $dataset_id
   *   Dataset id. Only needed for 'individual' type.
   * @param bool $private
   *   Whether the archive is private or not.
   *
   * @return array
   *   Array with 'directory', 'drupal_directory', and 'filename' keys.
   */
  protected function archivePathAndFilename(string $type, ?string $aggregation_type = '', string $theme_or_keyword = '', string $dataset_id = '', bool $private = FALSE): array {
    $date = Util::date()->format('Y-m-d');
    $base_path = $private ? $this->util->getDrupalPrivateFilesDirectory() : $this->util->getDrupalPublicFilesDirectory();
    $directory = Util::createArchiveFilePath($type, $aggregation_type, $theme_or_keyword, $dataset_id, $date);
    $filename = Util::createArchiveFilename($type, $aggregation_type, $theme_or_keyword, $dataset_id, $date, '');
    $full_directory_path = Util::adjustStorageLocation("{$base_path}/{$directory}", $private);
    $drupal_directory_path = Util::adjustStorageLocation("public://{$directory}", $private);
    $this->util->prepareDir($drupal_directory_path);

    return [
      'directory' => $full_directory_path,
      'drupal_directory' => $drupal_directory_path,
      'filename' => $filename,
    ];
  }

  /**
   * Return all downloadURLs from a single theme.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword.
   * @param bool $respectArchiveExclude
   *   Respect metadata archiveExclude if true. Defaults to false.
   *
   * @return array
   *   Array of downloadURLs.
   */
  public function getThemeDownloadUrls(string $theme_or_keyword, bool $respectArchiveExclude = FALSE) {
    // @todo This function may not be needed since we are not getting all
    // datasets for the theme, only the ones that recently changed.
    $archiveFolder = $this->util->getDrupalPublicFilesDir() . '/archive';
    $datasets = [];
    // Though most providers only have one theme, some like test_minimal have
    // more, so gather all datasets matching each of this provider's themes.
    // @todo Rework this to be about themes, not providers.
    foreach ((array) $theme_or_keyword as $theme) {
      // Pass human readable theme as search parameter.
      $datasets += $this->metastoreSearch->search([
        'theme' => $theme,
        'page' => 1,
        'page-size' => 300,
      ])->results;
      // @todo May not be the best place for this, but works for now.
      $this->util->prepareDir("{$archiveFolder}/{$theme}");
    }
    // @todo Rework this to get exclusion from config rather than provider.
    if ($respectArchiveExclude) {
      $datasets = array_filter($datasets, function ($dataset) {
        return !isset($dataset->archiveExclude) || !$dataset->archiveExclude;
      });
    }

    if (empty($datasets)) {
      return [];
    }

    // Gather and flatten all distributions, to get their downloadURL.
    return array_column(
      array_merge(array_column($datasets, 'distribution')),
      'downloadURL'
    );
  }

  /**
   * Return a csv's path and filename.
   *
   * @param string $downloadUrl
   *   Download URL from dataset.
   * @param bool $private
   *   Whether the archive is private or not.
   *
   * @return array
   *   Array with csv's path and filename values.
   */
  private function csvPathAndFilename(string $downloadUrl, bool $private): array {
    $csvFilename = basename($downloadUrl);
    // Check to see if its and archive path.
    if (str_contains($downloadUrl, 'dataset-archives/')) {
      // Its already in an archive path.
      $subdir = 'dataset-archives/';
      $pieces = explode($subdir, $downloadUrl);

    }
    else {
      $subdir = '';
      $pieces = explode('files/', $downloadUrl);
    }

    if (isset($pieces[1])) {
      $csvPathAndFilename = $pieces[1];
      $csvPath = str_replace("/{$csvFilename}", '', $csvPathAndFilename);
      if ($private) {
        $csvFullPath = $this->util->getDrupalPrivateFilesDirectory() . '/' . $subdir . $csvPath;
      }
      else {
        $csvFullPath = $this->util->getDrupalPublicFilesDir() . '/' . $subdir . $csvPath;
      }

      return [$csvFullPath, $csvFilename];
    }
    $this->logger->error('Download URL %downloadUrl is not valid. Could not be added to Zip file.', ['%downloadUrl' => $downloadUrl]);
    return [];
  }

  /**
   * Queue annual archives of full, keyword or theme aggregations.
   *
   * @param string $type
   *   The type of annual archive to create: individual or aggregate.
   * @param string|null $aggregate_of
   *   The aggregation type if applicable: 'keyword' or 'theme'.
   */
  public function queueAnnualArchives(string $type = 'individual', ?string $aggregate_of = NULL): void {
    $year = Util::date()->format('Y');
    // Load all the individual archives of the current year.
    $storage = $this->entityTypeManager->getStorage('dda_archive');
    $query = $storage->getQuery()
      ->condition('archive_type', $type, '=')
      ->condition('dataset_modified', "{$year}-01-01", '>=')
      ->condition('status', 1)
      // Doesn't matter who is running the query, we need them all.
      ->accessCheck(FALSE);
    if ($aggregate_of) {
      $query->condition('aggregate_of', $aggregate_of, '=');
    }
    $archive_ids = $query->execute();
    if (empty($archive_ids)) {
      $this->logger->info("No archives found for creating annual @type archives.", ['@type' => $type]);
      return;
    }
    $archives = $storage->loadMultiple($archive_ids);
    // Need to collate them if keyword or theme.
    $collated = $this->collateArchivesByType($archives, $type, $aggregate_of);

    // Loop through the collated and queue them.
    foreach ($collated as $term => $archives) {
      foreach ($archives as $privacy => $archive_infos) {
        $term_name = ($type === 'individual') ? '' : $term;
        $archive_ids = array_column($archive_infos, 'id');
        $private_bool = ($privacy === 'private');
        $this->addToAggregationQueue('annual', $term_name, $archive_ids, $private_bool, $aggregate_of);
      }
    }
  }

  /**
   * Collate archives by type.
   *
   * @param array $archives
   *   Array of DdaArchive entities.
   * @param string $type
   *   The type of archive: 'individual', 'aggregate', 'annual' to collate.
   * @param string|null $aggregate_of
   *   The aggregation of if applicable: 'keyword' or 'theme'.
   *
   * @return array
   *   Collated array of archives.
   */
  protected function collateArchivesByType(array $archives, string $type, ?string $aggregate_of): array {
    $collated = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchive $archive */
    foreach ($archives as $archive) {
      $private = $archive->isPrivate() ? 'private' : 'public';
      switch ($type) {
        case 'aggregate':
          $term = $archive->get('aggregate_on')->value;
          if (!empty($term) && !empty($aggregate_of)) {
            $archive_info = $this->createArchiveData($archive, $aggregate_of);
            (!isset($collated[$term][$private])) ? $collated[$term][$private] = [$archive_info] : $collated[$term][$private][$archive_info['id']] = $archive_info;
          }
          break;

        case 'individual':
          $archive_info = $this->createArchiveData($archive, $type);
          (!isset($collated[$type][$private])) ? $collated[$type][$private] = [$archive_info] : $collated[$type][$private][$archive_info['id']] = $archive_info;

          break;
      }
    }
    return $collated;
  }

  /**
   * Create an array of archive data from a DdaArchive entity.
   *
   * @param \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive
   *   The DdaArchive entity.
   * @param string $type
   *   The type of archive: 'individual', 'keyword', 'theme', 'annual_keyword',
   *   'annual_theme', 'annual', or 'current'.
   *
   * @return array
   *   Array of archive data.
   */
  protected function createArchiveData(DdaArchiveInterface $archive, string $type): array {
    return [
      'id' => $archive->id(),
      'type' => $type,
      'annual_create' => $year = Util::date()->format('Y'),
      'private' => $archive->isPrivate(),
      'dataset_id' => $archive->get('dataset_id')->value,
      'modified_date' => $archive->get('dataset_modified')->value,
      // @todo Assemble the files correctly local? remote?.
      'resource_files' => array_column($archive->getResourceFileItems(), 'uri'),
    ];
  }

  /**
   * Create an individual archive for a dataset.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data object.
   */
  public function createIndividualArchive(NodeInterface $data): void {
    if (empty($this->archiverSettings->get('archive')) || !$this->isArchiveWorthy($data)) {
      // Archiving is disabled or the dataset is not archive worthy so bail out.
      return;
    }

    $datetime_string = Util::grabMetadata($data, 'modified');
    $datetime_object = new DrupalDateTime($datetime_string, 'UTC');
    $datetime_object->setTimezone(new \DateTimeZone(DateTimeItemInterface::STORAGE_TIMEZONE));
    $formatted_modified_date = $datetime_object->format(DateTimeItemInterface::DATE_STORAGE_FORMAT);
    $dataset_id = Util::grabMetadata($data, 'identifier') ?? t('undefined');
    $file_size = 0;
    $archive_data = [
      'name' => Util::grabMetadata($data, 'title'),
      'archive_type' => 'individual',
      'access level' => Util::grabMetadata($data, 'accessLevel'),
      'dataset_modified' => $formatted_modified_date,
      'dataset_id' => $dataset_id,
      'themes' => $this->prepMultiValueFieldForStorage(Util::grabMetadata($data, 'theme'), 'theme'),
      'keywords' => $this->prepMultiValueFieldForStorage(Util::grabMetadata($data, 'keyword'), 'keyword'),
      'private' => Util::isDatasetPrivate($data),
      'distributions' => Util::grabMetadata($data, 'distribution') ?? [],
      'resource_files' => $this->prepMultiValueFileFieldForStorage($formatted_modified_date, $dataset_id, Util::grabMetadata($data, 'distribution'), $file_size, Util::isDatasetPrivate($data)),
      'size' => $file_size,
      'status' => '1',
    ];

    $storage = $this->entityTypeManager->getStorage('dda_archive');
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    $archive = $storage->create($archive_data);
    $archive->setSyncing(TRUE);
    $archive->save();
  }

  /**
   * Prepare a multi-value field for storage w/ look-up of entity references.
   *
   * @param array $values
   *   Array of values to prepare.
   * @param string $type
   *   The type of field, either 'theme', 'keyword', or ''.
   *
   * @return array
   *   Array formatted for storage.
   */
  public function prepMultiValueFieldForStorage(array $values, string $type = ''): array {
    $items = [];
    $reference_types = ['keyword', 'theme'];
    foreach ($values as $value) {
      if (Uuid::isValid($value) && in_array($type, $reference_types)) {
        // It is a uuid, so look it up.
        $referenced_entity = $this->getStorage($type)->retrieve($value, FALSE);
        if ($referenced_entity) {
          $value = json_decode($referenced_entity)->data ?? t('not found');
        }
      }
      // Map the value.
      $items[] = ['value' => $this->getMappedTerm($value)];
    }
    return $items;
  }

  /**
   * Prepare a multi-value File field for storage.
   *
   * @param string $modified_date
   *   The modified date of the dataset. Format Y-m-d.
   * @param string $dataset_id
   *   The identifier of the dataset.
   * @param array $values
   *   Array of values to prepare.
   * @param int $file_size
   *   The cumulative size of the files added, passed by reference.
   * @param bool $private
   *   TRUE if the files should be put in private storage.
   *
   * @return array
   *   Array of file entities.
   */
  public function prepMultiValueFileFieldForStorage(string $modified_date, string $dataset_id, array $values, int &$file_size, bool $private = FALSE): array {
    $items = [];
    $archive_path_and_filename = $this->archivePathAndFilename('individual', '', '', $dataset_id, $private);
    $destination_directory = $archive_path_and_filename['drupal_directory'];
    $destination_directory = Util::adjustStorageLocation($destination_directory, $private);
    foreach ($values as $resource) {
      if (Uuid::isValid($resource)) {
        // It is a uuid, so look it up.
        $referenced_entity = $this->getStorage('distribution')->retrieve($resource, FALSE);
        if ($referenced_entity) {
          $distribution_data = json_decode($referenced_entity)->data;
          $file_url = $distribution_data->downloadURL;
          $filename = basename(parse_url($file_url, PHP_URL_PATH));
          $filename = "{$dataset_id}_{$modified_date}_{$filename}";
          // Would be faster to trim the domain if it is a local file, but it
          // it might not be local, so we have to use the slow but sure way.
          $file_content = file_get_contents($file_url);
          $destination = "{$destination_directory}/{$filename}";
          $dictionary_url = $distribution_data->describedBy ?? '';
          $dictionary_filename = basename(parse_url($dictionary_url, PHP_URL_PATH));
          $dictionary_content = !empty($dictionary_url) ? file_get_contents($dictionary_url) : '';

          try {
            // Ensure the destination directory exists.
            $directory_exists = $this->fileSystem->prepareDirectory($destination_directory, FileSystemInterface::CREATE_DIRECTORY);
            if (!$directory_exists) {
              throw new \Exception("Failed to create the directory $destination_directory");
            }
            if (!$file_content) {
              throw new \Exception("Failed to get file content from $file_url");
            }
            if ($dictionary_url && !$dictionary_content) {
              throw new \Exception("Failed to get dictionary content from $dictionary_url");
            }

            // Save the data to the file and create a managed file entity.
            $managed_file = $this->fileRepository->writeData($file_content, $destination, FileExists::Replace);
            if ($managed_file instanceof FileInterface) {
              $items[] = ['target_id' => $managed_file->id()];
              $file_size += $managed_file->getSize();
              $this->logger->info('Created dataset archive file successfully: @file', ['@file' => $managed_file->getFileUri()]);
            }
            else {
              $this->logger->error('Failed to save archive file. @file', ['@file' => $destination]);
            }
            // Save the dictionary file if it exists.
            if ($dictionary_content) {
              $dictionary_destination = "{$destination_directory}/{$dictionary_filename}";
              $managed_dictionary_file = $this->fileRepository->writeData($dictionary_content, $dictionary_destination, FileExists::Replace);
              if ($managed_dictionary_file instanceof FileInterface) {
                $items[] = ['target_id' => $managed_dictionary_file->id()];
                $file_size += $managed_dictionary_file->getSize();
                $this->logger->info('Created dataset data dictionary archive file successfully: @file', ['@file' => $managed_dictionary_file->getFileUri()]);
              }
              else {
                $this->logger->error('Failed to save archive dictionary file. @file', ['@file' => $dictionary_destination]);
              }

            }
          }
          catch (\Exception $e) {
            $this->logger->error('Error saving archive file: @error', ['@error' => $e->getMessage()]);
          }
        }
      }
    }
    return $items;
  }

  /**
   * Check if this data is worthy of being having an archive.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data object.
   */
  public function isArchiveWorthy(NodeInterface $data): bool {
    if (!Util::isDataset($data) || !$data->isPublished()) {
      return FALSE;
    }
    // Check for private datasets.
    if ((!$this->archiverSettings->get('archive_private')) && (Util::isDatasetPrivate($data))) {
      return FALSE;
    }
    // @todo Check settings to see if private is allowed, and if this is private.
    if ($data->isNew() && !empty(Util::grabMetadata($data, 'distribution'))) {
      // It is new, and it has a resource(s).
      return TRUE;
    }
    else {
      return Util::archivableDataChanged($data,);
    }
  }

  /**
   * Get storage.
   *
   * @param string $schema_id
   *   The {schema_id} slug from the HTTP request.
   *
   * @return \Drupal\metastore\Storage\NodeData
   *   Entity storage.
   */
  protected function getStorage(string $schema_id): NodeData {
    if (!isset($this->storages[$schema_id])) {
      $this->storages[$schema_id] = $this->metastoreStorage->getInstance($schema_id);
    }
    return $this->storages[$schema_id];
  }

  /**
   * Get and set the term map from configuration.
   *
   * @return array
   *   Associative array of term map.
   *   Key is the original term, value is the mapped final term.
   */
  public function getMap(): array {
    if (empty($this->termMap)) {
      $map = $this->archiverSettings->get('theme_keyword_map');
      $raw_array = explode(PHP_EOL, $map);
      $term_map = [];
      foreach ($raw_array as $line) {
        $pieces = explode('->', $line);
        if (count($pieces) === 2) {
          $term_map[trim($pieces[0])] = trim($pieces[1]);
        }
      }
      $this->termMap = $term_map;
    }
    return $this->termMap;
  }

  /**
   * Get the mapped term from configuration, or return the original if no map.
   *
   * @param string $term
   *   The original term.
   *
   * @return string
   *   The mapped term, or the original if no map exists.
   */
  public function getMappedTerm(string $term): string {
    $map = $this->getMap();
    return $map[$term] ?? $term;
  }

  /**
   * Check if a term is blocked from archiving.
   *
   * @param string $type
   *   Either 'keyword' or 'theme'.
   * @param string $term
   *   The term to check.
   *
   * @return bool
   *   TRUE if the term is blocked, FALSE if not.
   */
  public function isBlockedTerm($type, $term): bool {
    $skip_type = "{$type}s_to_skip";
    $skip_map = $this->archiverSettings->get($skip_type) ?? [];
    $skip_map = explode(PHP_EOL, $skip_map);
    return in_array($term, $skip_map);
  }

  /**
   * Get dataset ids by theme or keyword.
   *
   * @param string $type
   *   Either 'keyword' or 'theme'.
   * @param string $theme_or_keyword
   *   The theme or keyword to search for.
   *
   * @return array
   *   Array of dataset ids for datasets that match the search. Separated
   *   by 'private' and 'public' keys.
   */
  public function getDatasetsIdsByThemeOrKeyword(string $type, string $theme_or_keyword): array {
    // Since themes or keywords can be mapped, we can not use a metastore
    // query directly. We have to load them and map them each.
    // This is not efficient or fast and the approach makes me sad.
    // @todo Rework this to be more efficient.
    $datasets = $this->getAllPublishedDatasets();
    $matched_dataset_ids = [];
    /** @var \Drupal\node\NodeInterface $dataset */
    foreach ($datasets as $dataset) {
      $terms = Util::grabMetadata($dataset, $type);
      $mapped_terms = array_map([$this, 'getMappedTerm'], $terms);
      if (in_array($theme_or_keyword, $mapped_terms)) {
        $private_text = Util::isDatasetPrivate($dataset) ? 'private' : 'public';
        $matched_dataset_ids[$private_text][] = $dataset->id();
      }
    }

    return $matched_dataset_ids;
  }

  /**
   * Get all published datasets, cached for subsequent calls.
   *
   * @return array
   *   Array of NodeInterface objects.
   */
  protected function getAllPublishedDatasets(): array {
    if (empty($this->publishedDatasets)) {
      $storage = $this->entityTypeManager->getStorage('node');
      $query = $storage->getQuery()
        ->condition('type', 'data')
        ->condition('field_data_type', 'dataset')
        ->condition('status', 1)
        // Doesn't matter who is running the query, we need them all.
        ->accessCheck(FALSE);
      $dataset_ids = $query->execute();
      $datasets = $storage->loadMultiple($dataset_ids);
      // Essentially caching these for subsequent calls.
      $this->publishedDatasets = $datasets;
    }

    return $this->publishedDatasets;
  }

}
