<?php

namespace Drupal\dkan_dataset_archiver\Service;

use Aws\Credentials\CredentialProvider;
use Aws\S3\S3Client;
use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;


/**
 * Aws S3 service.
 *
 * @package Drupal\dkan_dataset_archiver\Service
 */
class AwsS3Service {

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * S3 client.
   *
   * @var \Aws\S3\S3Client
   */
  private $client;

  /**
   * Constructs an AwsS3Service object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
    $this->logger = $logger;
  }

  /**
   * S3 client getter.
   *
   * @return \Aws\S3\S3Client
   *   S3 client.
   */
  public function getS3Client(): S3Client {
    if ($this->archiverSettings->get('remote_type') !== 'aws:s3' || empty($this->archiverSettings->get('remote_address')) || empty($this->archiverSettings->get('remote_region'))) {
      // Call it off.
      $this->logger->error('AWS S3 client requested but remote type is not aws:s3 or remote address/region is empty. Check your dkan_dataset_archiver settings.');
      throw new \RuntimeException('AWS S3 client requested but remote type is not aws:s3 or remote region or address is empty. Check your dkan_dataset_archiver settings.');
    }

    if (empty($this->client)) {
      $c = CredentialProvider::env();
      $composed = CredentialProvider::chain($c);
      /** @var \GuzzleHttp\Promise\Promise $promise */
      $promise = $composed();
      $awsCredentials = $promise->wait();
      $this->client = new S3Client([
        'version' => 'latest',
        'region' => $this->archiverSettings->get('remote_region'),
        'credentials' => $awsCredentials,
      ]);
      $this->client->registerStreamWrapper();
    }

    return $this->client;
  }

}
