<?php

namespace Drupal\Tests\dkan_dataset_archiver\Unit\Service;

use Drupal\common\DatasetInfo;
use Drupal\Core\DependencyInjection\Container;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Queue\QueueFactory;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\dkan_dataset_archiver\Service\dkan_dataset_archiverPurgeService;
use Drupal\dkan_dataset_archiver\Service\Util;
use Drupal\metastore_search\Search;
use Drupal\Tests\dkan_dataset_archiver\Traits\LoggerTestTrait;
use MockChain\Chain;
use MockChain\Options;
use PHPUnit\Framework\TestCase;

/**
 * Test for ArchiveService.
 */
class ArchiveServiceTest extends TestCase {

  use LoggerTestTrait;

  /**
   * Just a helper function to make creating directories easier.
   *
   * @param string $path
   *   Directory path to create.
   */
  public function createDir($path) {
    if (!is_dir($path)) {
      mkdir($path, 0777, TRUE);
    }
  }

  /**
   * Create and write to a file.
   *
   * @param string $path
   *   Path to file to create.
   * @param string $text
   *   Text to write to created file.
   */
  public function createAndWriteToFile($path, $text) {
    $myfile = fopen($path, "w");
    fwrite($myfile, $text);
    fclose($myfile);
  }

  /**
   * The setUp function.
   */
  protected function setUp(): void {
    $this->createDir('/tmp/archiveTestSrc/sites/default/files/');
    // Create a list of files and make them.
    $this->createAndWriteToFile('/tmp/archiveTestSrc/sites/default/files/thing1.txt', "Hello from Test 1");
    $this->createAndWriteToFile('/tmp/archiveTestSrc/sites/default/files/thing2.txt', "Hello from Test 2");

  }

  /**
   * The tearDown function.
   */
  protected function tearDown(): void {
    `rm -rf /tmp/archiveTestSrc/`;
  }

  /**
   * Return a mockChain container.
   *
   * @return \MockChain\Chain
   *   The mockChain container.
   */
  private function getContainer() {
    $options = (new Options())
      ->add('dkan.common.dataset_info', DatasetInfo::class)
      ->add('queue', QueueFactory::class)
      ->add('module_handler', ModuleHandler::class)
      ->add('dkan_dataset_archiver.util', Util::class)
      ->add('logger.channel.dkan_dataset_archiver', $this->getLoggerChain(TRUE)->getMock()->get('logger.channel.dkan_dataset_archiver'))
      ->index(0);

    $result = new \stdClass();
    $result->results = [
      ['distribution' => [['downloadURL' => '/tmp/archiveTestSrc/sites/default/files/thing1.txt']]],
      ['distribution' => [['downloadURL' => '/tmp/archiveTestSrc/sites/default/files/thing2.txt']]],
    ];

    $container = (new Chain($this))
      ->add(Container::class, 'get', $options)
      ->add(Search::class, 'search', $result)
      ->add(Util::class, 'getDrupalPublicFilesDir', '/tmp/archiveTestSrc/sites/default/files')
      ->add(Util::class, 'prepareDir', '')
      ->add(dkan_dataset_archiverPurgeService::class, 'clearUrls', '');

    return $container;
  }

  /**
   * Test for createIndividualZipByUuid.
   */
  public function testCreateIndividualZipByUuid() {
    $container = $this->getContainer();
    $container = $container->add(DatasetInfo::class, 'gather', [
      'latest_revision' => [
        'distributions' => [
          ['file_path' => '/tmp/archiveTestSrc/sites/default/files/thing1.txt'],
        ],
      ],
    ])
      ->getMock();
    \Drupal::setContainer($container);

    $archiveService = ArchiveService::create($container);
    $archiveService->createIndividualZipByUuid('test-asdf');
    $this->assertTrue(file_exists('/tmp/archiveTestSrc/sites/default/files/thing1.txt.zip'));
  }

  /**
   * Test for createIndividualZipByUuid when there's a published revision.
   */
  public function testCreateIndividualZipByUuidPublished() {
    $container = $this->getContainer();
    $container = $container->add(DatasetInfo::class, 'gather', [
      'latest_revision' => [
        'distributions' => [
          ['file_path' => '/tmp/archiveTestSrc/sites/default/files/thing1.txt'],
        ],
      ],
      'published_revision' => [
        'distributions' => [
          ['file_path' => '/tmp/archiveTestSrc/sites/default/files/thing2.txt'],
        ],
      ],
    ])
      ->getMock();
    \Drupal::setContainer($container);

    $archiveService = ArchiveService::create($container);
    $archiveService->createIndividualZipByUuid('test-asdf');
    $this->assertTrue(!file_exists('/tmp/archiveTestSrc/sites/default/files/thing1.txt.zip'));
    $this->assertTrue(file_exists('/tmp/archiveTestSrc/sites/default/files/thing2.txt.zip'));
  }

  /**
   * Test for createIndividualZipByUuid when uuid is not found.
   */
  public function testCreateIndividualZipByUuidNotFound() {
    $container = $this->getContainer();
    $container = $container->add(DatasetInfo::class, 'gather', [
      'notice' => [],
    ])
      ->getMock();
    \Drupal::setContainer($container);

    $archiveService = ArchiveService::create($container);
    $archiveService->createIndividualZipByUuid('test-asdf');
    $lastLoggerMessage = $this->loggerChain->getStoredInput('notices');
    $this->assertEquals('Uuid not found.', $lastLoggerMessage[0]);
  }

  /**
   * Test for createIndividualZips.
   */
  public function testCreateIndividualZips() {

    $container = $this->getContainer()->getMock();
    \Drupal::setContainer($container);
    $archiveService = ArchiveService::create($container);
    $themes_and_keywords = [
      'test_minimal',
    ];

    $archiveService->createIndividualZips($mockProvider);
    $this->assertTrue(file_exists('/tmp/archiveTestSrc/sites/default/files/thing1.txt.zip'));
    $this->assertTrue(file_exists('/tmp/archiveTestSrc/sites/default/files/thing2.txt.zip'));
  }

}
