<?php

namespace Drupal\dkan_dataset_archiver;

use Drupal\common\JsonResponseTrait;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableJsonResponse;

/**
 * Wraps DKAN's JsonResponseTrait to add tag-based caching.
 */
trait CachedJsonResponseTrait {
  use JsonResponseTrait;

  /**
   * Get the cache for a particular tag, if any.
   *
   * @param string $cacheTag
   *   This cache tag.
   *
   * @return object|bool
   *   The cache object found, FALSE otherwise.
   */
  protected function getCacheTag(string $cacheTag): object|bool {
    return $this->cacheBackend->get($cacheTag);
  }

  /**
   * Get cached response.
   *
   * @param object $cacheObject
   *   The cache object.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   The cacheable json response.
   */
  protected function getCachedResponse(object $cacheObject): CacheableJsonResponse {
    $response = new CacheableJsonResponse($cacheObject->data);
    if (isset($cacheObject->tags)) {
      $response->getCacheableMetadata()->setCacheTags($cacheObject->tags);
    }
    return $response;
  }

  /**
   * Set cache, then get response.
   *
   * @param string $tag
   *   This cache tag.
   * @param mixed $data
   *   Cache data.
   * @param array $cacheTags
   *   Additional cache tags.
   * @param int $lifespan
   *   Lifespan of the cache, in seconds. Use Cache::PERMANENT for no expiry.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   The cacheable json response.
   */
  protected function setCacheGetResponse(string $tag, $data, array $cacheTags = [], int $lifespan = Cache::PERMANENT): CacheableJsonResponse {
    if ($lifespan !== Cache::PERMANENT) {
      $lifespan = $lifespan + time();
    }
    $this->cacheBackend->set(
      $tag,
      $data,
      $lifespan,
      $cacheTags,
    );

    $response = new CacheableJsonResponse($data, 200, []);
    $response->getCacheableMetadata()->setCacheTags($cacheTags);
    return $response;
  }

}
