<?php

namespace Drupal\dkan_dataset_archiver\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for DdaArchive edit forms.
 *
 * @ingroup dda_archive
 */
class DdaArchiveForm extends ContentEntityForm {

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * Constructs a new DdaArchiveForm.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user account.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $bundle_info,
    TimeInterface $time,
    AccountProxyInterface $account,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($entity_repository, $bundle_info, $time);
    $this->account = $account;
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('current_user'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $this->alterFormBasedOnSettings($form);

    $form['dataset_id']['#states'] = [
      // @todo this required state is not working.
      'required' => [
        ':input[name="archive_type"]' => ['value' => 'individual'],
      ],
      'visible' => [
        ':input[name="archive_type"]' => ['value' => 'individual'],
      ],
    ];

    $visible_on_aggregate = [
      'visible' => [
        ':input[name="archive_type"]' => [
          ['value' => 'aggregate'],
          ['value' => 'annual'],
          ['value' => 'current'],
        ],
      ],
    ];
    $form['source_archives']['#states'] = $visible_on_aggregate;
    $form['remote_url']['#states'] = $visible_on_aggregate;
    $form['aggregate_of']['#states'] = $visible_on_aggregate;
    $form['aggregate_on']['#states'] = $visible_on_aggregate;

    $form['themes']['#states'] = [
      'visible' => [
        ':input[name="archive_type"]' => ['value' => 'individual'],
      ],
    ];
    $form['keywords']['#states'] = [
      'visible' => [
        ':input[name="archive_type"]' => ['value' => 'individual'],
      ],
    ];

    // Authoring information for administrators.
    if (isset($form['user_id'])) {
      $form['author'] = [
        '#type' => 'details',
        '#title' => $this->t('Authoring information'),
        '#group' => 'advanced',
        '#weight' => -3,
        '#optional' => TRUE,
      ];
    }

    $form['user_id']['#group'] = 'author';
    $form['created']['#group'] = 'author';
    $form['changed']['#group'] = 'author';
    $form['files'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Related files'),
      '#weight' => 35,
    ];

    $form['resource_files']['#group'] = 'files';
    $form['resource_files']['#upload_location'] = 'public://dataset-archives';

    // @todo Hide remote_url if settings do not allow remote archives.
    $form['remote_url']['#group'] = 'files';
    $form['size']['#group'] = 'files';

    $form['footer'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Status'),
      '#weight' => 99,
    ];

    $form['status']['#group'] = 'footer';
    // Make revisions default on.
    $form["revision"]["#default_value"] = TRUE;
    $form["revision_information"]["#open"] = TRUE;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    $archive = $this->entity;

    // Save as a new revision if requested to do so.
    if ($form_state->getValue('revision') == TRUE) {
      $archive->setNewRevision();

      // If a new revision is created, save the current user as revision author.
      $archive->setRevisionCreationTime($this->time->getRequestTime());
      $archive->setRevisionUserId($this->account->id());
    }
    else {
      $archive->setNewRevision(FALSE);
    }

    $status = parent::save($form, $form_state);

    switch ($status) {
      case SAVED_NEW:

        $this->messenger()->addMessage($this->t("Created the archive '%label'.", [
          '%label' => $archive->getName(),
        ]));
        return SAVED_NEW;

      default:
        $this->messenger()->addMessage($this->t("Updated the archive '%label'.", [
          '%label' => $archive->getName(),
        ]));
        return SAVED_UPDATED;
    }
  }

  /**
   * Alter the form based on settings.
   *
   * @param array $form
   *   The form by reference.
   */
  protected function alterFormBasedOnSettings(array &$form): void {
    if (!$this->archiverSettings->get('archive_by_theme')) {
      unset($form['aggregate_of']['widget']['#options']['theme']);
    }
    if (!$this->archiverSettings->get('archive_by_keyword')) {
      unset($form['aggregate_of']['widget']['#options']['keyword']);
    }
    if (!$this->archiverSettings->get('archive_by_theme') && !$this->archiverSettings->get('archive_by_keyword')) {
      // Can't aggregate if neither is enabled.
      unset($form['archive_type']['widget']['#options']['aggregate']);
      $form['aggregate_of']['#access'] = FALSE;
      $form['aggregate_on']['#access'] = FALSE;
    }
    if (!$this->archiverSettings->get('create_annual_archives')) {
      unset($form['archive_type']['widget']['#options']['annual']);
    }
    if (!$this->archiverSettings->get('create_current_download')) {
      unset($form['archive_type']['widget']['#options']['current']);
    }
    if ($this->archiverSettings->get('storage_locations') === 'local') {
      unset($form['remote_url']);
    }
  }

}
