<?php

namespace Drupal\dkan_dataset_archiver;

use Drupal\common\JsonResponseTrait;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableJsonResponse;

/**
 * Wraps DKAN's JsonResponseTrait to add tag-based caching.
 */
trait CachedJsonResponseTrait {
  // Only used by ValidationController and static content generation.
  use JsonResponseTrait;

  /**
   * Get the cache for a particular tag, if any.
   *
   * @param string $cacheTag
   *   This cache tag.
   *
   * @return object|bool
   *   The cache object found, FALSE otherwise.
   */
  protected function getCacheTag(string $cacheTag): object|bool {
    return $this->cacheBackend->get($cacheTag);
  }

  /**
   * Get cached response.
   *
   * @param object $cacheObject
   *   The cache object.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   The cacheable json response.
   */
  protected function getCachedResponse(object $cacheObject): CacheableJsonResponse {
    $response = new CacheableJsonResponse($cacheObject->data);
    if (isset($cacheObject->tags)) {
      $response->getCacheableMetadata()->setCacheTags($cacheObject->tags);
    }
    return $response;
  }

  /**
   * Set cache, then get response.
   *
   * @param string $tag
   *   This cache tag.
   * @param mixed $data
   *   Cache data.
   * @param array $cacheTags
   *   Additional cache tags.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   The cacheable json response.
   */
  protected function setCacheGetResponse(string $tag, $data, array $cacheTags = []): CacheableJsonResponse {
    $this->cacheBackend->set(
      $tag,
      $data,
      Cache::PERMANENT,
      $cacheTags,
    );

    $response = new CacheableJsonResponse($data, 200, []);
    $response->getCacheableMetadata()->setCacheTags($cacheTags);
    return $response;
  }

}
