<?php

namespace Drupal\dkan_dataset_archiver\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\metastore\LifeCycle\LifeCycle;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscriber to manage related events.
 */
class Subscriber implements EventSubscriberInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver logger channel.
   * @param \Drupal\dkan_dataset_archiver\Service\ArchiveService $archiveService
   *   The archive service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, LoggerInterface $logger, ArchiveService $archiveService) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
    $this->archiveService = $archiveService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('dkan_dataset_archiver.archive_service')
    );
  }

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[LifeCycle::EVENT_DATASET_UPDATE][] = ['updateArchiveQueue', 60];

    return $events;
  }

  /**
   * Update a provider's monthly and annual archives.
   *
   * @param \Drupal\metastore\LifeCycle\LifeCycle $event
   *   Provider publication event.
   */
  public function updateArchiveQueue(LifeCycle $event): void {
    // @todo get the theme or keywords from the dataset.
    $metadata = $event->getMetaData();
    // @todo check configuration to see what we need to act on keywords|themes.
    // @todo work out the rest of the logic.
    $themes = $metadata->get('field_theme')->getValue();
    $keywords = $metadata->get('field_keywords')->getValue();
    $this->archiveService->scheduleArchiving($keywords ?? $themes, TRUE);
  }

}
