<?php

namespace Drupal\dkan_dataset_archiver\Service;

use Drupal\Core\File\FileSystem;
use Drupal\taxonomy\Entity\Term;

/**
 * A collection of utility functions.
 */
class Util {

  /**
   * Make a string machine friendly.
   *
   * @param string $string
   *   String to machinize.
   *
   * @return string
   *   Machinized string.
   */
  public static function machinize($string): string {
    $new = preg_replace('/[^A-Za-z0-9 ]/', '', $string);
    $size = strlen($new);
    if ($size > 40) {
      $excess = $size - 40;
      $new = substr($new, $excess);
    }

    return $new;
  }

  /**
   * Get the path to the public files directory.
   *
   * @return string
   *   Path to the public files directory.
   */
  public static function getDrupalPublicFilesDirectory(): string {
    return \Drupal::service('file_system')->realpath('public://');
  }

  /**
   * Verify a directory exists, or create it.
   *
   * @param string $directory
   *   Directory.
   *
   * @return bool
   *   TRUE if it exists (or was created) and is writable. FALSE otherwise.
   */
  public static function prepareDirectory(string $directory) : bool {
    /** @var \Drupal\Core\File\FileSystem $fs */
    $fs = \Drupal::service('file_system');
    $flags = FileSystem::CREATE_DIRECTORY | FileSystem::MODIFY_PERMISSIONS;
    return $fs->prepareDirectory($directory, $flags);
  }

  /**
   * Use instead of PHP's date() to consistently get date in CMS' time zone.
   *
   * @param string $date
   *   String date to set or modify, defaults to 'now'.
   *
   * @return \DateTime
   *   Current date time .
   */
  public static function date($date = 'now'): \DateTime {
    return new \DateTime($date, new \DateTimeZone('America/New_York'));
  }

  /**
   * A wrapper around the static getDrupalPublicFilesDirectory function.
   *
   * @return string
   *   Path to the public files directory.
   */
  public function getDrupalPublicFilesDir() {
    return self::getDrupalPublicFilesDirectory();
  }

  /**
   * A wrapper around the static prepareDirectory function.
   *
   * @param string $directory
   *   Path to directory to prepare.
   *
   * @return bool
   *   TRUE if it exists (or was created) and is writable. FALSE otherwise.
   */
  public function prepareDir(string $directory) {
    return self::prepareDirectory($directory);
  }

  /**
   * Wraps static Term creation function so it can be mocked in phpunit.
   *
   * @param string $termName
   *   The name of the term.
   * @param string $taxonomyName
   *   The vocabulary the term belongs to.
   *
   * @return \Drupal\taxonomy\Entity\Term
   *   The metastore search api service.
   */
  public function createTaxonomyTerm(string $termName, string $taxonomyName): Term {
    return Term::create([
      'name' => $termName,
      'vid' => $taxonomyName,
    ]);
  }

}
