<?php

namespace Drupal\dkan_dataset_archiver\Service;

use Drupal\common\DatasetInfo;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Queue\QueueFactory;
use Drupal\dkan_dataset_archiver\AwsS3Trait;
use Drupal\dkan_dataset_archiver\HelperTrait;
use Drupal\metastore_search\Search;
use Procrastinator\Result;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;

/**
 * Archive Service.
 */
class ArchiveService implements ContainerInjectionInterface {
  // @todo Figure out if this needs to handle purging removed in migration.
  use AwsS3Trait;
  use HelperTrait;

  const YEARS_OF_ARCHIVES = 7;
  const README = 'readme.txt';
  // @todo Pull this from config remote_address.
  const ARCHIVE_BUCKET = 'this was an S3 address';

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Dkan datasetInfo.
   *
   * @var \Drupal\common\DatasetInfo
   */
  private $datasetInfo;

  /**
   * The metastore search api service.
   *
   * @var \Drupal\metastore_search\Search
   */
  private $metastoreSearch;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * Utility of helper functions. @todo rework this.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\Util
   */
  protected $util;

  /**
   * Current year.
   *
   * @var string
   */
  private $year;

  /**
   * {@inheritDoc}
   *
   * @param \Drupal\common\DatasetInfo $datasetInfo
   *   Dkan datasetInfo.
   * @param \Drupal\Core\Queue\QueueFactory $queue
   *   The queue factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver logger channel.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler.
   * @param \Drupal\dkan_dataset_archiver\Service\Util $util
   *   Utility.
   * @param \Drupal\metastore_search\Search $metastoreSearchService
   *   Metastore Search wrapper for the SearchApi.
   */
  public function __construct(
    DatasetInfo $datasetInfo,
    QueueFactory $queue,
    LoggerInterface $logger,
    ModuleHandler $moduleHandler,
    Util $util,
    Search $metastoreSearchService,
  ) {
    $this->datasetInfo = $datasetInfo;
    $this->queue = $queue;
    $this->moduleHandler = $moduleHandler;
    $this->util = $util;
    $this->year = Util::date()->format('Y');
    $this->logger = $logger;
    $this->metastoreSearch = $metastoreSearchService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('dkan.common.dataset_info'),
      $container->get('queue'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('module_handler'),
      $container->get('dkan_dataset_archiver.util'),
      $container->get('dkan.metastore_search.service'),
    );
  }

  /**
   * Bring content from S3's archive.
   */
  public function fetch(): void {
    /** @var \Procrastinator\Result $result */
    $result = $this->retrieveYearsFromS3Bucket(self::ARCHIVE_BUCKET);

    $level = $result->getStatus() == Result::ERROR ? LogLevel::ERROR : LogLevel::INFO;
    $message = $result->getStatus() == Result::ERROR ? $result->getError() : 'All files were retrieved';
    $this->logger->log($level, $message, []);
  }

  /**
   * Retrieve years from S3 bucket.
   *
   * @param string $bucketName
   *   The S3 bucket name.
   *
   * @return mixed
   *   The result of the operation. @todo Investigate and make more specific.
   */
  private function retrieveYearsFromS3Bucket($bucketName) {
    $this->s3Bucket = $bucketName;

    $fileSystem = $this->getAwsS3Filesystem();

    $contents = $fileSystem->listContents('', TRUE);
    $previous_year = (int) Util::date('-1 year')->format('Y');
    $oldest_year = $previous_year - self::YEARS_OF_ARCHIVES;

    $filePaths = array_values(array_filter(array_map(function ($thing) use ($bucketName, $oldest_year) {
      $info = [];
      if ($thing['type'] == 'file' && $this->archiveYearAllowed($thing['dirname'], $oldest_year)) {
        $info['path'] = "s3://{$bucketName}/" . $thing['path'];
        $info['directory'] = $thing['dirname'];
      }
      return $info;
    }, $contents), function ($item) {
      return !empty($item);
    }));

    foreach ($filePaths as $info) {
      $directory = $this->util->getDrupalPublicFilesDir() . '/archive/' . $info['directory'];
      if (!file_exists($directory)) {
        mkdir($directory, 0777, TRUE);
      }
      $fileFetcher = $this->getFileFetcher($info['path'], $directory);
      $result = $fileFetcher->run();
      if ($result->getStatus() == Result::ERROR) {
        return $result;
      }
    }

    $result = new Result();
    $result->setStatus(Result::DONE);
    return $result;
  }

  /**
   * Ensure archive year is allowed.
   *
   * @param string $directory
   *   Directory containing the archive.
   * @param int $oldest_year
   *   Oldest year allowed.
   *
   * @return bool
   *   TRUE if the year is within the allowed range.
   */
  private function archiveYearAllowed(string $directory, int $oldest_year): bool {
    if (empty($directory)) {
      return FALSE;
    }
    $pieces = explode('/', $directory);
    if (empty($pieces)) {
      return FALSE;
    }
    $year = end($pieces);
    return is_numeric($year) && (int) $year > $oldest_year;
  }

  /**
   * Schedule the archiving of a theme's current data now or later.
   *
   * @param string $theme_name
   *   The name of the theme to archive. @todo Rework away from provider.
   * @param bool $later
   *   Update happens now, if TRUE, or later if FALSE (default).
   */
  public function scheduleArchiving(string $theme_name, bool $later = FALSE): void {
    // @todo Set month and year here once, and pass as additional parameters.
    if ($later) {
      /** @var \Drupal\Core\Queue\QueueInterface $archiveQueue */
      $archiveQueue = $this->queue->get('archive_provider');
      $queueId = $archiveQueue->createItem($theme_name);

      $this->logger->notice(
        'Updating current archives for %provider has been queued: queue_id %queueId', [
          '%provider' => $theme_name,
          '%queueId' => $queueId,
        ]
      );
    }
    else {
      $this->runArchiving($theme_name);
    }
  }

  /**
   * Runs archiving operations on a provider.
   *
   * @param string $theme_or_keyword
   *   Provider.
   */
  public function runArchiving(string $theme_or_keyword): void {
    // @todo check if theme or keyword should be skipped.
    $skip_it = [];
    if (!in_array($theme_or_keyword, $skip_it)) {
      $this->createMonthlyArchive($theme_or_keyword);
      $this->createAnnualArchive($theme_or_keyword);
      // @todo fix routes.
      $archive_api_routes = [
        'api/1/archive/topics/archive',
        'api/1/archive/topics/current-zip',
      ];
      $cache_tags = ['topics_archives', 'topics_current_zips'];
      // @todo rework this without purge service.
      // ->archivePurgeService->clearUrls($archive_api_routes, $cache_tags);
    }
    $this->createDownloadAll($theme_or_keyword);
  }

  /**
   * Create a theme or keyword's individual zips.
   *
   * @param string $theme_or_keyword
   *   The theme or keyword to aggregate.
   * @param bool $later
   *   Creation happens now, if TRUE, or later if FALSE (default).
   */
  public function createIndividualZips(string $theme_or_keyword, bool $later = FALSE): void {
    $downloadUrls = $this->getThemeDownloadUrls($theme_or_keyword);
    $id = "@todo fix me";
    // @todo Skip test themes.
    if (substr($id, 0, 5) !== 'test_') {
      foreach ($downloadUrls as $url) {
        if ($later) {
          /** @var \Drupal\Core\Queue\QueueInterface $individualZipsQueue */
          $individualZipsQueue = $this->queue->get('individual_zips');
          $queueId = $individualZipsQueue->createItem($url);
          $this->logger->notice(
            'Individual zip for %url has been queued: queue_id %queueId', [
              '%url' => $url,
              '%queueId' => $queueId,
            ]
          );
        }
        else {
          $this->createIndividualZip($url);
        }
      }
    }
  }

  /**
   * Create an individual zip for a given url.
   *
   * @param string $url
   *   The url to create a zip for.
   */
  public function createIndividualZip($url): void {
    if ($url) {
      $publicFiles = $this->util->getDrupalPublicFilesDir();
      $cleanUrl = explode("sites/default/files/", $url)[1];
      $fileSrc = "{$publicFiles}/{$cleanUrl}";
      $fileDest = $fileSrc . '.zip';
      $filename = explode('/', $cleanUrl);
      $filename = end($filename);

      $zip = new \ZipArchive();
      $zip->open($fileDest, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
      $zip->addFile($fileSrc, $filename);
      $zip->close();
      $this->logger->notice(
        "Zip for %fileDest created.", [
          '%fileDest' => $cleanUrl,
        ]
      );
    }
  }

  /**
   * Create an individual zip by uuid.
   *
   * @param string $uuid
   *   The uuid of the dataset to zip.
   */
  public function createIndividualZipByUuid(string $uuid): void {
    $datasetInfo = $this->datasetInfo->gather($uuid);
    if (!isset($datasetInfo['notice'])) {
      $revision = $datasetInfo['latest_revision'];
      if (isset($datasetInfo['published_revision'])) {
        $revision = $datasetInfo['published_revision'];
      }
      if (count($revision['distributions'])) {
        foreach ($revision['distributions'] as $distribution) {
          $this->createIndividualZip($distribution['file_path']);
        }
      }
    }
    else {
      $this->logger->notice("Uuid not found.");
    }
  }

  /**
   * Create a $theme_or_keyword's monthly archive.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword.
   */
  private function createMonthlyArchive(string $theme_or_keyword): void {
    $theme = $theme_or_keyword;

    [$zipPath, $zipFilename] = $this->archivePathAndFilename($theme, TRUE);
    $zip = new \ZipArchive();
    $zip->open("{$zipPath}/{$zipFilename}", \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

    $downloadUrls = $this->getThemeDownloadUrls($theme_or_keyword, TRUE);
    $this->addDownloadUrlFilesToZip($downloadUrls, $zip);
    $this->addOtherFilesToMonthlyArchive($theme_or_keyword, $zip);

    $this->logger->notice(
      'Created monthly archive for %theme in %path.', [
        '%theme' => $theme,
        '%path' => $zipPath,
      ]
    );

    $this->queueBackupToS3($theme_or_keyword, (int) $this->year, "{$zipPath}/{$zipFilename}");
  }

  /**
   * Create a theme_or_keyword's download-all zip.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword. @todo Rework this to be about Theme or keyword.
   */
  public function createDownloadAll(string $theme_or_keyword): void {
    $publicFolder = $this->util->getDrupalPublicFilesDir();
    $zipPath = $publicFolder . "/archive/{$theme_or_keyword}/current";
    $this->util->prepareDir($zipPath);

    $machine_name = strtolower(str_replace(' ', '_', $theme_or_keyword));
    $zipFilename = "{$machine_name}_current_data.zip";

    $zip = new \ZipArchive();
    $zip->open("{$zipPath}/{$zipFilename}", \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

    $downloadUrls = $this->getThemeDownloadUrls($theme_or_keyword);
    $this->addDownloadUrlFilesToZip($downloadUrls, $zip);
    $this->addOtherFilesToMonthlyArchive($theme_or_keyword, $zip);

    $this->logger->notice(
      'Created download-all zip for %theme in %path.', [
        '%theme' => $theme_or_keyword,
        '%path' => $zipPath,
      ]
    );
  }

  /**
   * Add files from a list of urls to a zip archive.
   *
   * @param array $downloadUrls
   *   List of CSV download urls.
   * @param \ZipArchive $zip
   *   Zip archive.
   */
  private function addDownloadUrlFilesToZip(array $downloadUrls, \ZipArchive $zip): void {
    foreach ($downloadUrls as $downloadUrl) {
      [$csvPath, $csvFilename] = $this->csvPathAndFilename($downloadUrl);
      if (isset($csvPath) && isset($csvFilename)) {
        $zip->addFile("{$csvPath}/{$csvFilename}", $csvFilename);
      }
    }
  }

  /**
   * Create a theme_or_keyword's annual archive.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword @todo rework this.
   */
  private function createAnnualArchive(string $theme_or_keyword): void {
    $theme = $theme_or_keyword;
    $theme_machine_name = strtolower(str_replace(' ', '_', $theme));
    $archivePath = $this->archivePathAndFilename($theme)[0];
    $archiveName = "{$theme_machine_name}_{$this->year}.zip";

    $monthly_archives = (new Finder())
      ->files()
      ->name('/.*\d{2}_\d{4}.zip/')
      ->in($archivePath)
      ->depth('== 0');

    $zip = new \ZipArchive();
    $zip->open("{$archivePath}/{$archiveName}", \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
    foreach ($monthly_archives as $monthly_archive) {
      $zip->addFile($monthly_archive->getRealPath(), $monthly_archive->getBasename());
    }
    $zip->close();

    $this->logger->notice(
      'Created %year annual archive for %theme.', [
        '%year' => $this->year,
        '%theme' => $theme,
      ]
    );

    $this->queueBackupToS3($theme_or_keyword, (int) $this->year, "{$archivePath}/{$archiveName}");
  }

  /**
   * Queue the copying of a new archive to its theme_or_keyword's folder in S3.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword @todo rework this.
   * @param int $year
   *   Year.
   * @param string $filePath
   *   Filename including full path.
   */
  private function queueBackupToS3(string $theme_or_keyword, int $year, string $filePath): void {

    if ($this->skipBackup($theme_or_keyword)) {
      return;
    }

    /** @var \Drupal\Core\Queue\QueueInterface $backupQueue */
    $backupQueue = $this->queue->get('backup_to_s3');
    $backupQueue->createItem([
      'provider' => $theme_or_keyword,
      'year' => $year,
      'filePath' => $filePath,
    ]);
  }

  /**
   * Copy an archive to S3.
   *
   * @param array $data
   *   Associative array with 'provider', 'year' and 'filePath' named keys.
   */
  public function backupToS3(array $data): void {
    ['theme_or_keyword' => $theme_or_keyword, 'year' => $year, 'filePath' => $filePath] = $data;

    // Check again if this 'backup_to_s3' queue item was in a database copied.
    if ($this->skipBackup($theme_or_keyword)) {
      return;
    }

    $destination = 's3://' . self::ARCHIVE_BUCKET . "/{$theme_or_keyword}/{$year}/" . basename($filePath);

    // Register the stream wrapper, if not already set.
    $fileSystem = $this->getAwsS3Filesystem();

    $stream = fopen($destination, 'w');
    // @todo set to 10_000_000 once our site runs PHP 7.4
    stream_set_chunk_size($stream, 10 * 1000 * 1000);
    copy($filePath, $destination);
  }

  /**
   * Check to see if this is exempt from backup.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword @todo rework this.
   *
   * @return bool
   *   TRUE to skip, FALSE not to.
   */
  public static function skipBackup(string $theme_or_keyword) {
    // @todo Get the skip from config.
    if ('prod' !== getenv('AH_SITE_ENVIRONMENT') && 'test_minimal' !== $theme_or_keyword) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Add other files to the monthly archive.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword @todo rework this.
   * @param \ZipArchive $zip
   *   Archive.
   */
  private function addOtherFilesToMonthlyArchive(string $theme_or_keyword, \ZipArchive $zip): void {
    // @todo Rework this to add json manifest.
    $this->addReadmeTxtToMonthlyArchive($zip);
    $this->addDictionariesToMonthlyArchive($theme_or_keyword, $zip);
  }

  /**
   * Add the readme.txt to the monthly archive.
   *
   * @param \ZipArchive $zip
   *   Archive.
   */
  private function addReadmeTxtToMonthlyArchive(\ZipArchive $zip): void {
    $modulePath = $this->moduleHandler->getModule('dkan_dataset_archiver')->getPath();

    $readmePath = $modulePath . '/files/' . self::README;
    $zip->addFile($readmePath, self::README);
  }

  /**
   * Add a provider's dictionaries to its monthly archive.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword @todo needs reworking.
   * @param \ZipArchive $zip
   *   Archive.
   */
  private function addDictionariesToMonthlyArchive(string $theme_or_keyword, \ZipArchive $zip): void {
    // @todo This needs major reworking because dictionary must be extendable.
    // Maybe default check for existence of *dictionary.pdf and if not grab from
    // dkan in some way.
    $dictionaries = [];

    foreach ($dictionaries as $dictionary) {
      $zip->addFile($dictionary['realPath'], $dictionary['filename']);
    }
    $zip->close();
  }

  /**
   * Return an archive's path and filename.
   *
   * @param string $theme
   *   Provider theme.
   * @param bool $deleteOldFilename
   *   Whether or not to remove the old filename format, defaults to FALSE.
   *
   * @return array
   *   Array with zip archive's path and filename values.
   */
  private function archivePathAndFilename(string $theme, bool $deleteOldFilename = FALSE): array {
    $month = Util::date()->format('m');

    $publicFiles = $this->util->getDrupalPublicFilesDir();
    $directory = "{$publicFiles}/archive/{$theme}/{$this->year}";
    $this->util->prepareDir($directory);
    $machine_name = strtolower(str_replace(' ', '_', $theme));

    // @todo Remove in the month following deployment Might not need this.
    if ($deleteOldFilename) {
      // Filename was "{$machine_name}_archive_{$month}_{$year}.zip".
      // Delete it to avoid adding 2 differently-named archives for same month.
      $oldFilename = "{$machine_name}_archive_{$month}_{$this->year}.zip";
      $toDelete = "{$directory}/{$oldFilename}";
      if (file_exists($toDelete)) {
        unlink($toDelete);
      }
    }

    $filename = "{$machine_name}_{$month}_{$this->year}.zip";

    return [$directory, $filename];
  }

  /**
   * Return all downloadURLs from a single theme.
   *
   * @param string $theme_or_keyword
   *   Theme or keyword. @todo This entire function may not be needed.
   * @param bool $respectArchiveExclude
   *   Respect metadata archiveExclude if true. Defaults to false.
   *
   * @return array
   *   Array of downloadURLs.
   */
  private function getThemeDownloadUrls(string $theme_or_keyword, bool $respectArchiveExclude = FALSE) {
    // @todo This function may not be needed since we are no getting all
    // datasets for the theme, only the ones that recently changed.
    $archiveFolder = $this->util->getDrupalPublicFilesDir() . '/archive';
    $datasets = [];
    // Though most providers only have one theme, some like test_minimal have
    // more, so gather all datasets matching each of this provider's themes.
    // @todo Rework this to be about themes, not providers.
    foreach ((array) $theme_or_keyword as $theme) {
      // Pass human readable theme as search parameter.
      $datasets += $this->metastoreSearch->search([
        'theme' => $theme,
        'page' => 1,
        'page-size' => 300,
      ])->results;
      // @todo May not be the best place for this, but works for now.
      $this->util->prepareDir("{$archiveFolder}/{$theme}");
    }
    // @todo Rework this to get exclusion from config rather than provider.
    if ($respectArchiveExclude) {
      $datasets = array_filter($datasets, function ($dataset) {
        return !isset($dataset->archiveExclude) || !$dataset->archiveExclude;
      });
    }

    if (empty($datasets)) {
      return [];
    }

    // Gather and flatten all distributions, to get their downloadURL.
    return array_column(
      array_merge(...array_column($datasets, 'distribution')),
      'downloadURL'
    );
  }

  /**
   * Return a csv's path and filename.
   *
   * @param string $downloadUrl
   *   Download URL from dataset.
   *
   * @return array
   *   Array with csv's path and filename values.
   */
  private function csvPathAndFilename(string $downloadUrl): array {
    $csvFilename = basename($downloadUrl);
    $pieces = explode('sites/default/files', $downloadUrl);
    if (isset($pieces[1])) {
      $csvPathAndFilename = explode('sites/default/files', $downloadUrl)[1];
      $csvPath = str_replace("/{$csvFilename}", '', $csvPathAndFilename);
      $csvFullPath = $this->util->getDrupalPublicFilesDir() . $csvPath;
      return [$csvFullPath, $csvFilename];
    }
    return [];
  }

  /**
   * Create annual archives for every providers.
   */
  public function createAnnualArchives(): void {
    $themes = (new Finder())
      ->directories()
      ->in(DRUPAL_ROOT . '/../src/site/files/archive')
      ->depth('== 0');

    foreach ($themes as $theme) {
      $years = (new Finder())
        ->directories()
        ->in(DRUPAL_ROOT . '/../src/site/files/archive/' . $theme->getFilename())
        ->depth('== 0');

      foreach ($years as $year) {
        $this->createAnnualThemeArchive($theme->getBasename(), $year);
      }
    }
  }

  /**
   * Helper to create an annual archive for a theme.
   *
   * @param string $theme
   *   Theme.
   * @param \Symfony\Component\Finder\SplFileInfo $year
   *   A year directory within a theme directory.
   */
  private function createAnnualThemeArchive(string $theme, SplFileInfo $year): void {
    $theme_machine_name = strtolower(str_replace(' ', '_', $theme));
    $monthly_archives = (new Finder())
      ->files()
      ->name('/.*\d{2}_\d{4}.zip/')
      ->in($year->getRealPath())
      ->depth('== 0');
    $zip = new \ZipArchive();
    $zipFilename = "{$year->getRealPath()}/{$theme_machine_name}_{$year->getBasename()}.zip";
    $zip->open($zipFilename, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
    foreach ($monthly_archives as $monthly_archive) {
      $zip->addFile($monthly_archive->getRealPath(), $monthly_archive->getBasename());
    }
    $zip->close();
  }

}
