<?php

namespace Drupal\dkan_dataset_archiver\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\dkan_dataset_archiver\CachedJsonResponseTrait;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\dkan_dataset_archiver\Service\Util;
use League\Flysystem\Adapter\Local;
use League\Flysystem\Filesystem;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Archive controller.
 */
class ArchiveController implements ContainerInjectionInterface {

  use CachedJsonResponseTrait;

  /**
   * The filesystem object.
   *
   * @var \League\Flysystem\Filesystem
   */
  private $filesystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManager
   */
  private $streamWrapperManager;

  /**
   * Cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend.
   */
  public function __construct(CacheBackendInterface $cacheBackend) {
    $this->cacheBackend = $cacheBackend;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ArchiveController {
    return new static(
      $container->get('cache.default'),
      $container->get('stream_wrapper_manager'),
    );
  }

  /**
   * Return info about all archives, structured by topic and year.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   This this is not clear what this returns. @todo Investigate and fix.
   */
  public function archive(): Response {

    // Check for Topics' archives in existing cache.
    if ($cachedTopicsArchives = $this->getCacheTag('topics_archives')) {
      return $this->getCachedResponse($cachedTopicsArchives);
    }

    $structure = $this->getStructuredContentOfDirectory('archive');

    // @todo Technically, the archive pages should have cache dependency on
    // their topic because they display the footnote field. In reality these
    // change infrequently.
    return $this->setCacheGetResponse('topics_archives', $structure);
  }

  /**
   * Get structured content of a directory.
   *
   * @param string $directory
   *   The directory to get content from.
   *
   * @return array
   *   The structured content.
   */
  private function getStructuredContentOfDirectory(string $directory): array {
    $content = $this->getFilesystem()->listContents($directory, TRUE);
    return $this->structuredContent($content);
  }

  /**
   * Structure content.
   *
   * @param array $content
   *   Array of files and folders.
   *
   * @return array
   *   Content structured by topic, then year.
   */
  private function structuredContent(array $content): array {
    $flatStructure = array_values(array_filter(array_map(function ($item) {
      return $this->setInfo($item);
    }, array_filter($content, function ($item) {
      return $item['type'] == 'file';
    })), function ($item) {
      $yearSet = isset($item['year']);
      $yearAllowed = $yearSet && $this->yearAllowed($item['year']);
      $topicSet = isset($item['topic']);
      return $yearSet && $yearAllowed && $topicSet;
    }));

    $structure = [];
    foreach ($flatStructure as $file) {
      $topic = $file['topic'];
      unset($file['topic']);
      $year = $file['year'];
      unset($file['year']);

      if (!isset($structure[$topic])) {
        $structure[$topic] = [];
      }

      if (!isset($structure[$topic][$year])) {
        $structure[$topic][$year] = [];
      }

      $structure[$topic][$year][] = $file;
    }

    return $structure;
  }

  /**
   * Restrict the number of years of archive returned.
   */
  private function yearAllowed(string $year): bool {
    $previous_year = (int) Util::date('-1 year')->format('Y');
    $oldest_year = $previous_year - ArchiveService::YEARS_OF_ARCHIVES;
    return ((int) $year) > $oldest_year;
  }

  /**
   * Set item info.
   *
   * @param array $item
   *   Info on one file or folder.
   *
   * @return array
   *   Modified info.
   */
  private function setInfo(array $item): array {
    $info = [];
    $filePath = "public://{$item['path']}";
    $info['url'] = $this->fileCreateUrl($filePath);
    $info['size'] = $item['size'];
    [$month, $year] = $this->getMonthAndYearFromFilename($item['filename']);
    $info['type'] = isset($month) ? 'Monthly' : 'Annual';

    if (isset($month)) {
      $info['month'] = $month;
      $info['day'] = date('d', $item['timestamp']);
    }

    $info['year'] = $year;

    $info['topic'] = $this->getTopicFromDirectoryName($item['dirname']);

    return $info;
  }

  /**
   * Set the filesystem object.
   *
   * @param \League\Flysystem\Filesystem $filesystem
   *   The filesystem object.
   */
  public function setFileSystem(Filesystem $filesystem): void {
    $this->filesystem = $filesystem;
  }

  /**
   * Extract the month MM (if any) and year YYYY from end of filename.
   *
   * @param string $filename
   *   The filename.
   *
   * @return array
   *   Array with month (or NULL) and year (or NULL).
   */
  private function getMonthAndYearFromFilename(string $filename): array {
    if (preg_match('/_(\d\d)_(\d{4})$/', $filename, $matches)) {
      return [$matches[1], $matches[2]];
    }
    if (preg_match('/_(\d{4})$/', $filename, $matches)) {
      return [NULL, $matches[1]];
    }
    return [NULL, NULL];
  }

  /**
   * Extract the topic from the directory name.
   *
   * @param string $directoryName
   *   The directory name.
   *
   * @return string|null
   *   The topic, or NULL if it cannot be determined.
   */
  private function getTopicFromDirectoryName(string $directoryName): ?string {
    $pieces = explode('/', $directoryName);
    return $pieces[1] ?? NULL;
  }

  /**
   * Create a URL for a file URI.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return string
   *   The file URL on the public side of the site.
   */
  private function fileCreateUrl($uri): string {
    if ($wrapper = $this->streamWrapperManager->getViaUri($uri)) {
      $url = $wrapper->getExternalUrl();
      // @todo We need a solution to do this for all environments to convert
      // private edit domain urls to public.
      // Used to be dkan_dataset_archiver_force_url_to_public().
      return $url;
    }
    return $uri;
  }

  /**
   * Get (or create) the filesystem object.
   *
   * @return \League\Flysystem\Filesystem
   *   The filesystem object.
   */
  private function getFilesystem(): Filesystem {
    if (is_null($this->filesystem)) {
      $this->filesystem = new Filesystem(new Local(Util::getDrupalPublicFilesDirectory()));
    }
    return $this->filesystem;
  }

  /**
   * Return info about a topic's most current zip file.
   *
   * @param string $topic
   *   Human-readable topic.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about a topic's current zip.
   */
  public function topicCurrentZip(string $topic): Response {
    $cacheTag = "topic:{$topic}:zip:current";
    // Check for that topic's current zip in existing cache.
    if ($cachedTopicCurrentZip = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedTopicCurrentZip);
    }
    $latest = $this->latestZip('archive/' . $topic);

    return $this->setCacheGetResponse($cacheTag, $latest);
  }

  /**
   * Return info about every topic's most current zip file.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about every topic's current zip.
   */
  public function allCurrentZips(): Response {
    // Check for that topic's current zip in existing cache.
    if ($cachedTopicCurrentZip = $this->getCacheTag('topics_current_zips')) {
      return $this->getCachedResponse($cachedTopicCurrentZip);
    }
    $latest = $this->latestZip('archive');

    return $this->setCacheGetResponse('topics_current_zips', $latest);
  }

  /**
   * Return info about the most current zip file in a directory.
   *
   * @param string $directory
   *   Directory to check.
   *
   * @return array
   *   Array of info about the most current zip file in the directory.
   */
  private function latestZip(string $directory): array {
    $contents = $this->getFilesystem()->listContents($directory, TRUE);
    $structure = [];

    $items = array_filter($contents, function ($item) {
      $dirFragments = explode('/', $item['dirname']);
      return 'current' === end($dirFragments);
    });

    $date = Util::date();
    $year = $date->format('Y');
    $month = $date->format('m');

    foreach ($items as $item) {
      $dirFragments = explode('/', $item['dirname']);
      if (isset($dirFragments[1])) {
        $structure[$dirFragments[1]][$year] = [
          'url' => $this->fileCreateUrl("public://{$item['path']}"),
          'size' => $item['size'],
          'type' => 'Monthly',
          'month' => $month,
        ];
      }
    }

    return $structure;
  }

}
