<?php

namespace Drupal\Tests\dkan_dataset_archiver\Unit\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\File\FileSystem as DrupalFileSystem;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Drupal\dkan_dataset_archiver\Controller\ArchiveController;
use Drupal\dkan_dataset_archiver\Service\Util;
use Drupal\Tests\dkan_dataset_archiver\Traits\LoggerTestTrait;
use League\Flysystem\Filesystem as LeagueFilesystem;
use MockChain\Chain;
use MockChain\Options;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Test for ArchiveController.
 */
class ArchiveControllerTest extends TestCase {
  use LoggerTestTrait;

  /**
   * Test the archive response.
   */
  public function testArchiveResponse() {

    $container = $this->getContainer()->getMock();
    \Drupal::setContainer($container);

    $archiveController = ArchiveController::create($container);
    $jsonResponse = $archiveController->archive();

    $result = json_decode($jsonResponse->getContent());

    // Archives older than 7 years aren't included.
    $this->assertFalse(isset($result->Hospitals->{'2013'}), 'Datasets from 2013 should not be here.');
    $this->assertFalse(isset($result->Hospitals->{'2017'}), 'Datasets from 2017 should not be here.');
    $this->assertTrue(isset($result->Hospitals->{'2021'}), 'Dataset from 2021 should be here.');
    $this->assertTrue(isset($result->{'Nursing homes including rehab services'}->{'2020'}), 'Dataset from 2020 should be here.');
    $this->assertTrue(isset($result->{'Nursing homes including rehab services'}->{'2021'}), 'Dataset from 2021 should be here.');
  }

  /**
   * Test the cached response of the archive.
   */
  public function testArchiveCachedResponse() {

    $container = $this->getContainer()
      ->add(CacheBackendInterface::class, 'get', (object) ['data' => 'foo']);

    $archiveController = ArchiveController::create($container->getMock());
    $cachedResponse = $archiveController->archive();

    $this->assertEquals('foo', json_decode($cachedResponse->getContent()));
  }

  /**
   * Test the current zip for a topic.
   */
  public function testTopicCurrentZip() {

    $container = $this->getContainer()->getMock();
    \Drupal::setContainer($container);

    $archiveController = ArchiveController::create($container);
    $jsonResponse = $archiveController->topicCurrentZip('Hospitals');

    $result = json_decode($jsonResponse->getContent());

    $date = Util::date();
    $currentYear = $date->format('Y');
    $currentMonth = $date->format('m');
    // Only 2021 Hospitals is included.
    $this->assertFalse(isset($result->{'Nursing homes including rehab services'}));
    $this->assertFalse(isset($result->Hospitals->{'2020'}));
    $this->assertEquals($result->Hospitals->{$currentYear}->month, $currentMonth);
    $this->assertEquals('Monthly', $result->Hospitals->{$currentYear}->type);
  }

  /**
   * Test the current zips cached response for a topic.
   */
  public function testTopicCurrentZipCachedResponse() {

    $container = $this->getContainer()
      ->add(CacheBackendInterface::class, 'get', (object) ['data' => 'foo']);

    $archiveController = ArchiveController::create($container->getMock());
    $cachedResponse = $archiveController->topicCurrentZip('dialysis');

    $this->assertEquals('foo', json_decode($cachedResponse->getContent()));
  }

  /**
   * Test the current zips.
   *
   * @todo Make it non specific.
   */
  public function testAllCurrentZips() {

    $container = $this->getContainer()->getMock();
    \Drupal::setContainer($container);

    $archiveController = ArchiveController::create($container);
    $jsonResponse = $archiveController->allCurrentZips();

    $result = json_decode($jsonResponse->getContent());

    $date = Util::date();
    $currentYear = $date->format('Y');
    $currentMonth = $date->format('m');
    // Only latest year is included.
    $this->assertFalse(isset($result->Hospitals->{'2020'}));
    $this->assertFalse(isset($result->{'Nursing homes including rehab services'}->{'2017'}));
    $this->assertEquals($result->Hospitals->{$currentYear}->month, $currentMonth);
    $this->assertEquals('Monthly', $result->Hospitals->{$currentYear}->type);
    $this->assertEquals($currentMonth, $result->{'Nursing homes including rehab services'}->{$currentYear}->month);
    $this->assertEquals('Monthly', $result->{'Nursing homes including rehab services'}->{$currentYear}->type);
  }

  /**
   * Test all current zips cached response.
   */
  public function testAllCurrentZipsCachedResponse() {

    $container = $this->getContainer()
      ->add(CacheBackendInterface::class, 'get', (object) ['data' => 'foo']);

    $archiveController = ArchiveController::create($container->getMock());
    $cachedResponse = $archiveController->allCurrentZips();

    $this->assertEquals('foo', json_decode($cachedResponse->getContent()));
  }

  /**
   * Test of the file system.
   */
  public function testSetFileSystem() {

    $container = $this->getContainer()->getMock();
    \Drupal::setContainer($container);
    $mockFileSystem = (new Chain($this))
      ->add(LeagueFilesystem::class)
      ->getMock();

    $archiveController = ArchiveController::create($container);
    $archiveController->setFileSystem($mockFileSystem);

    $this->assertTrue(TRUE);
  }

  /**
   * Get a mock container with commonly used services.
   *
   * @return \MockChain\Chain
   *   The mock container.
   */
  private function getContainer(): Chain {
    $options = (new Options())
      ->add('file_system', DrupalFileSystem::class)
      ->add('stream_wrapper_manager', StreamWrapperManager::class)
      ->add('cache.default', CacheBackendInterface::class)
      ->add('logger.channel.dkan_dataset_archiver', $this->getLoggerChain(TRUE)->getMock()->get('logger.channel.dkan_dataset_archiver'))
      ->index(0);

    return (new Chain($this))
      ->add(ContainerInterface::class, 'get', $options)
      ->add(DrupalFileSystem::class, 'realPath', __DIR__ . '/../../../files')
      ->add(CacheBackendInterface::class, 'get', FALSE);
  }

}
