<?php

namespace Drupal\dkan_dataset_archiver\Controller;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This is the controller for the archives.
 */
class DdaArchiveController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new DdaArchiveController.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(DateFormatterInterface $date_formatter, RendererInterface $renderer) {
    $this->dateFormatter = $date_formatter;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('date.formatter'),
      $container->get('renderer')
    );
  }

  /**
   * Displays a DdaArchive revision.
   *
   * @param int $dda_archive_revision
   *   The DdaArchive revision ID.
   *
   * @return array
   *   An array suitable for drupal_render().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function revisionShow(int $dda_archive_revision): array {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $storage */
    $storage = $this->entityTypeManager()->getStorage('dda_archive');
    $dda_archive = $storage->loadRevision($dda_archive_revision);
    return $this->entityTypeManager()
      ->getViewBuilder('dda_archive')
      ->view($dda_archive);
  }

  /**
   * Page title callback for a DdaArchive revision.
   *
   * @param int $dda_archive_revision
   *   The DdaArchive revision ID.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The page title.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function revisionPageTitle(int $dda_archive_revision): TranslatableMarkup {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $storage */
    $storage = $this->entityTypeManager()->getStorage('dda_archive');
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $dda_archive */
    $dda_archive = $storage->loadRevision($dda_archive_revision);
    return $this->t('Revision of %title from %date', [
      '%title' => $dda_archive->label(),
      '%date' => $this->dateFormatter->format($dda_archive->getRevisionCreationTime()),
    ]);
  }

  /**
   * Generates an overview table of older revisions of a DdaArchive.
   *
   * @param \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $dda_archive
   *   A DdaArchive object.
   *
   * @return array
   *   An array as expected by drupal_render().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function revisionOverview(DdaArchiveInterface $dda_archive): array {
    $account = $this->currentUser();
    $dda_archive_storage = $this->entityTypeManager()->getStorage('dda_archive');

    $langcode = $dda_archive->language()->getId();
    $langname = $dda_archive->language()->getName();
    $languages = $dda_archive->getTranslationLanguages();
    $has_translations = (count($languages) > 1);
    $translation_text = $this->t('@langname revisions for %title', [
      '@langname' => $langname,
      '%title' => $dda_archive->label(),
    ]);
    $no_translation_text = $this->t('Revisions for %title', ['%title' => $dda_archive->label()]);
    $build['#title'] = $has_translations ? $translation_text : $no_translation_text;

    $header = [$this->t('Revision'), $this->t('Operations')];
    $revert_permission = ($account->hasPermission("manage dataset archive revisions"));
    $delete_permission = ($account->hasPermission("manage dataset archive revisions"));

    $rows = [];

    $vids = array_column($dda_archive_storage->getAggregateQuery()
      ->allRevisions()
      ->condition('id', $dda_archive->id())
      ->groupBy('vid')
      ->accessCheck(TRUE)
      ->execute(), 'vid');

    $latest_revision = TRUE;

    foreach (array_reverse($vids) as $vid) {
      /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $revision */
      $revision = $dda_archive_storage instanceof RevisionableStorageInterface ? $dda_archive_storage->loadRevision($vid) : NULL;
      // Only show revisions that are affected by the language that is being
      // displayed.
      if ($revision->hasTranslation($langcode)) {
        $username = [
          '#theme' => 'username',
          '#account' => $revision->getRevisionUser(),
        ];

        $row = [];
        $column = [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
            '#context' => [
              'date' => Link::fromTextAndUrl(
                $this->dateFormatter->format($revision->getRevisionCreationTime(), 'short'),
                new Url('entity.dda_archive.revision', [
                  'dda_archive' => $dda_archive->id(),
                  'dda_archive_revision' => $vid,
                ])
              )->toString(),
              'username' => $this->renderer->render($username),
              'message' => [
                '#markup' => $revision->getRevisionLogMessage(),
                '#allowed_tags' => Xss::getHtmlTagList(),
              ],
            ],
          ],
        ];
        $row[] = $column;

        if ($latest_revision) {
          $row[] = [
            'data' => [
              '#prefix' => '<em>',
              '#markup' => $this->t('Current revision'),
              '#suffix' => '</em>',
            ],
          ];
          foreach ($row as &$current) {
            $current['class'] = ['revision-current'];
          }
          unset($current);
          $latest_revision = FALSE;
        }
        else {
          $links = [];
          if ($revert_permission) {
            $links['revert'] = [
              'title' => $this->t('Revert'),
              'url' => $has_translations ?
              Url::fromRoute('entity.dda_archive.translation_revert', [
                'dda_archive' => $dda_archive->id(),
                'dda_archive_revision' => $vid,
                'langcode' => $langcode,
              ]) :
              Url::fromRoute('entity.dda_archive.revision_revert', [
                'dda_archive' => $dda_archive->id(),
                'dda_archive_revision' => $vid,
              ]),
            ];
          }

          if ($delete_permission) {
            $links['delete'] = [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('entity.dda_archive.revision_delete', [
                'dda_archive' => $dda_archive->id(),
                'dda_archive_revision' => $vid,
              ]),
            ];
          }

          $row[] = [
            'data' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }

        $rows[] = $row;
      }
    }

    $build['dda_archive_revisions_table'] = [
      '#theme' => 'table',
      '#rows' => $rows,
      '#header' => $header,
    ];

    return $build;
  }

}
