<?php

namespace Drupal\dkan_dataset_archiver\EventSubscriber;

use Drupal\common\Events\Event;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\State\StateInterface;
use Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\metastore\LifeCycle\LifeCycle;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscriber to manage related events.
 */
class Subscriber implements EventSubscriberInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[LifeCycle::EVENT_DATASET_UPDATE][] = ['createIndividualArchive', 60];
    $events[ArchivePostSaveEvent::EVENT_NAME][] = ['addArchiveToAggregatorState', 0];
    $events[ArchivePostSaveEvent::EVENT_NAME][] = ['updateExistingArchives', 10];

    return $events;
  }

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver logger channel.
   * @param \Drupal\dkan_dataset_archiver\Service\ArchiveService $archiveService
   *   The archive service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
    ArchiveService $archiveService,
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
    $this->archiveService = $archiveService;
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('dkan_dataset_archiver.archive_service'),
      $container->get('config.factory'),
      $container->get('state'),
    );
  }

  /**
   * Update existing archives when a dataset is updated.
   *
   * @param \Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent $event
   *   The archive post save event.
   */
  public function updateExistingArchives(ArchivePostSaveEvent $event): void {
    // @todo this will detect if an individual archive has been updated
    // (not initial save) and then update any aggregate archives that already
    // include the archive.
    // Update the manifest in the zip files.
    // Alter the updated date on the keyword, theme or annual archives.
  }

  /**
   * Update an individual dataset archive.
   *
   * @param \Drupal\common\Events\Event $event
   *   Dataset update event.
   */
  public function createIndividualArchive(Event $event): void {
    if ($this->archiverSettings->get('archive') !== '1') {
      // Archiving is turned off, so bail out.
      return;
    }
    /**  @var \Drupal\metastore\NodeWrapper\Data $metastore */
    $metastore = $event->getData();
    /**  @var \Drupal\node\NodeInterface $dataset */
    $dataset = $metastore->getEntity();
    $this->archiveService->createIndividualArchive($dataset);
  }

  /**
   * Set the last archive save time, keywords and themes in state.
   *
   * @param \Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent $event
   *   The archive post save event.
   */
  public function addArchiveToAggregatorState(ArchivePostSaveEvent $event): void {
    $archive_active = (bool) $this->archiverSettings->get('archive');
    $archive = $event->getArchive();
    $type = $archive->getArchiveType();
    $aggregate_by_keyword = (bool) $this->archiverSettings->get('archive_by_keyword');
    $aggregate_by_theme = (bool) $this->archiverSettings->get('archive_by_theme');
    if (($type === 'individual') && ($archive_active) && ($aggregate_by_keyword || $aggregate_by_theme)) {
      // We should be aggregating.
      $template = [
        'time' => time(),
        'keyword' => [],
        'theme' => [],
      ];
      $aggregation_info = $this->state->get('dkan_dataset_archiver.aggregations', $template) ?? $template;

      if ($aggregate_by_keyword) {
        $keywords = $archive->get('keywords');
        $aggregation_info = $this->injectItems('keyword', $keywords, $archive->id(), $aggregation_info);
      }

      if ($aggregate_by_theme) {
        $themes = $archive->get('themes');
        $aggregation_info = $this->injectItems('theme', $themes, $archive->id(), $aggregation_info);
      }
      $this->state->set('dkan_dataset_archiver.aggregations', $aggregation_info);
    }
  }

  /**
   * Inject items into the aggregation info array.
   *
   * @param string $type
   *   The type of items, either 'keywords' or 'themes'.
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The items to inject.
   * @param int $archive_id
   *   The ID of the archive.
   * @param array $aggregation_info
   *   The current aggregation info array.
   *
   * @return array
   *   The updated aggregation info array.
   */
  protected function injectItems(string $type, FieldItemListInterface $items, int $archive_id, array $aggregation_info): array {
    foreach ($items as $item) {
      /** @var \Drupal\Core\Field\FieldItemInterface $item */
      $term = $item->value;
      $term = $this->archiveService->getMappedTerm($term);
      if (!$this->archiveService->isBlockedTerm($type, $term)) {
        $aggregation_info[$type][$term][] = $archive_id;
      }
    }
    return $aggregation_info;
  }

}
