<?php

namespace Drupal\dkan_dataset_archiver\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Alt Text Validation Setting form.
 */
class ArchiverSettings extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'dkan_dataset_archiver_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['dkan_dataset_archiver.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('dkan_dataset_archiver.settings');
    $form['#title'] = $this->t('DKAN Dataset Archiver Settings');

    $form['archive'] = [
      '#type' => 'radios',
      '#title' => $this->t('Archive datasets when updated, deleted, or unpublished.'),
      '#default_value' => $config->get('archive') ?? 0,
      '#options' => [
        '1' => $this->t('On'),
        '0' => $this->t('Off'),
      ],
    ];

    $form['datasets_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude specific datasets.'),
      '#description' => $this->t('Enter dataset IDs to exclude from archiving (one per line).'),
      '#default_value' => $config->get('datasets_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    $form['collation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('How to group archived datasets'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['collation']['archive_by_theme'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset theme'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Theme when updated, deleted, or unpublished during the same "Aggregation delay" window.'),
      '#default_value' => $config->get('archive_by_theme') ?? FALSE,
    ];

    $form['collation']['themes_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude by theme.'),
      '#description' => $this->t('Enter themes to exclude from archiving (one per line).'),
      '#default_value' => $config->get('themes_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['archive_by_keyword'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive datasets by dataset keyword'),
      '#description' => $this->t('Select this if you want to group your archives by dataset Keywords when updated, deleted, or unpublished during the same "Aggregation delay" window.'),
      '#default_value' => $config->get('archive_by_keyword') ?? FALSE,
    ];

    $form['collation']['keywords_to_skip'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude by keyword.'),
      '#description' => $this->t('Enter keywords to exclude from archiving (one per line).'),
      '#default_value' => $config->get('keywords_to_skip'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['create_annual_archives'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create annual archives of all public datasets.'),
      '#description' => $this->t('At the end of each year, a zip file will generate of all datasets for that year. If you’ve also selected “Archive by theme” or “Archive by keyword,” separate annual archives will be created for those as well.'),
      '#default_value' => $config->get('create_annual_archives') ?? FALSE,
    ];

    $form['archive_private'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Archive private datasets.'),
      '#description' => $this->t('Creates separate archives for private datasets in addition to the public ones the users has chosen to generate. If you’ve also selected “Archive by theme” or “Archive by keyword,” private datasets will be grouped and archived the same way as public datasets.'),
      '#default_value' => $config->get('archive_private') ?? 0,
      '#states' => [
        'visible' => [
          [':input[name="archive"]' => ['value' => 1]],
        ],
      ],
    ];

    $form['collation']['aggregation_delay'] = [
      '#title' => $this->t('Aggregation delay'),
      '#description' => $this->t('The number of minutes to wait after a dataset is updated before creating an aggregated archive. If datasets bulk update automatically, this can be 10 minutes, if datasets update by hand, this should be a larger number to prevent premature aggregation. Minimum is 10 minutes, maximum is 1440 minutes (24 hours).'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 4,
      '#max' => 1440,
      '#min' => 10,
      '#default_value' => $config->get('aggregation_delay') ?? 10,
      '#states' => [
        'visible' => [
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['collation']['theme_keyword_map'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Map keywords or themes to other words.'),
      '#description' => $this->t("Enter keywords or themes that should be changed 'Original name -> New name' (one pair per line)."),
      '#default_value' => $config->get('theme_keyword_map'),
      '#rows' => 7,
      '#cols' => 30,
      '#resizeable' => 'both',
      '#states' => [
        'visible' => [
          [':input[name="archive_by_keyword"]' => ['checked' => TRUE]],
          [':input[name="archive_by_theme"]' => ['checked' => TRUE]],
        ],
      ],
    ];

    $form['locations_group'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Archive Storage and Retention'),
      '#open' => TRUE,
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['storage_locations'] = [
      '#type' => 'radios',
      '#title' => $this->t('Storage location'),
      '#description' => $this->t('Choose where you would like to store the dataset archives.'),
      '#default_value' => $config->get('storage_locations') ?? 0,
      '#options' => [
        'local' => $this->t('Create only local archives %example', ['%example' => '(files/dkan_dataset_archives/)']),
        'remote' => $this->t('Create only remote archives'),
        'local_and_remote' => $this->t('Create archives in both local and remote locations'),
      ],
      '#states' => [
        // Show this field only if the archiver is on.
        'visible' => [
          ':input[name="archive"]' => [
            ['value' => '1'],
          ],
        ],
      ],
    ];

    $form['locations_group']['local_years_retained'] = [
      '#prefix' => '<div class="container-inline">',
      '#title' => $this->t('Number of years to retain local archives'),
      '#description' => $this->t('Set to 0 to retain indefinitely.'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 3,
      '#max' => 365,
      '#min' => 0,
      '#default_value' => $config->get('local_years_retained') ?? 0,
      '#suffix' => '</div>',
      '#states' => [
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'local'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    $form['locations_group']['remote_years_retained'] = [
      '#prefix' => '<div class="container-inline">',
      '#title' => $this->t('Number of years to retain remote archives'),
      '#description' => $this->t('Set to 0 to retain indefinitely.'),
      '#type' => 'number',
      '#attributes' => [
        'data-type' => 'number',
      ],
      '#size' => 3,
      '#max' => 365,
      '#min' => 0,
      '#default_value' => $config->get('remote_years_retained') ?? 0,
      '#suffix' => '</div>',
      '#states' => [
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    $form['locations_group']['remote_address'] = [
      '#title' => $this->t('The AWS S3 bucket address'),
      '#description' => $this->t('Enter the full S3 bucket address including s3:// and the bucket name. Example: s3://my-bucket-name-identifier'),
      '#type' => 'textfield',
      '#size' => 80,
      '#maxlength' => 2000,
      '#default_value' => $config->get('remote_address') ?? '',
      '#states' => [
        'required' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
        // Show this textfield only if the box is selected above.
        'visible' => [
          ':input[name="storage_locations"]' => [
            ['value' => 'remote'],
            ['value' => 'local_and_remote'],
          ],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory()->getEditable('dkan_dataset_archiver.settings');
    $config->set('archive', $form_state->getValue('archive'))
      ->set('archive_by_theme', $form_state->getValue('archive_by_theme'))
      ->set('archive_by_keyword', $form_state->getValue('archive_by_keyword'))
      ->set('create_annual_archives', $form_state->getValue('create_annual_archives'))
      ->set('archive_private', $form_state->getValue('archive_private'))
      ->set('aggregation_delay', $form_state->getValue('aggregation_delay'))
      ->set('storage_locations', $form_state->getValue('storage_locations'))
      ->set('local_years_retained', $form_state->getValue('local_years_retained'))
      ->set('remote_years_retained', $form_state->getValue('remote_years_retained'))
      ->set('remote_address', rtrim($form_state->getValue('remote_address'), '/ '))
      ->set('themes_to_skip', $this->convertSort($form_state->getValue('themes_to_skip')))
      ->set('keywords_to_skip', $this->convertSort($form_state->getValue('keywords_to_skip')))
      ->set('datasets_to_skip', $this->convertSort($form_state->getValue('datasets_to_skip')))
      ->set('theme_keyword_map', $this->convertSortArrowFormat($form_state->getValue('theme_keyword_map')))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Dedupe, sort and convert to one per line.
   *
   * @param string $multi_values_raw
   *   A comma or new line separated string of multiple values.
   *
   * @return string
   *   A new-line separated string that has been deduped and sorted.
   */
  private function convertSort($multi_values_raw): string {
    // Clean up any bad data by converting all separators into ','.
    $multi_values = str_replace(["\r\n", "\n", "\r", ', ', ' ,'], ',', $multi_values_raw);
    $values = explode(',', $multi_values);
    // Dedupe and sort.
    $values = array_unique($values);
    natcasesort($values);
    $multi_values_sorted = implode("\n", $values);
    return $multi_values_sorted;
  }

  /**
   * Dedupe, sort and convert while sanitizing the arrows.
   *
   * @param string $multi_values_raw
   *   A comma or new line separated string of multiple values.
   *
   * @return string
   *   A new-line separated string that has been deduped and sorted.
   */
  private function convertSortArrowFormat($multi_values_raw): string {
    // Clean up any bad data by converting all separators into ','.
    $multi_values = str_replace(["\r\n", "\n", "\r", ', ', ' ,'], ',', $multi_values_raw);
    $values = explode(',', $multi_values);
    // Dedupe and clean.
    foreach ($values as &$value) {
      $parts = explode('->', $value);
      if (count($parts) === 2) {
        $value = trim($parts[0]) . ' -> ' . trim($parts[1]);
        if (empty(trim($parts[0])) || empty(trim($parts[1]))) {
          // One side is missing so remove it.
          $value = '';
        }
      }
      else {
        // If it doesn't have exactly one arrow, remove it to avoid problems.
        $value = '';
      }
    }
    // Must unset the reference.
    unset($value);
    $values = array_unique($values);
    natcasesort($values);
    $multi_values_sorted = implode("\n", $values);
    return $multi_values_sorted;
  }

}
