<?php

namespace Drupal\dkan_dataset_archiver\Service;

use Drupal\Core\File\FileSystem;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\Entity\Term;

/**
 * A collection of utility functions.
 */
class Util {

  /**
   * Check if any of the archivable data changed.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data object.
   *
   * @return bool
   *   TRUE if archivable data changed, FALSE if not.
   */
  public static function archivableDataChanged(NodeInterface $data): bool {
    if (Util::isDataset($data)) {
      if (
        Util::datasetFieldChanged($data, 'title') ||
        Util::datasetFieldChanged($data, 'modified') ||
        Util::datasetFieldChanged($data, 'distribution')
        ) {
        // Consider this changed since one the three things tracked by an
        // archive changed (title, modified or distribution).
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Build an array of entity reference targets from an array of ids.
   *
   * @param array $ids
   *   The entity ids.
   *
   * @return array
   *   An array targets that can be handed to an entity reference field.
   */
  public static function buildEntityReferenceTargets(array $ids): array {
    $targets = [];
    foreach ($ids as $id) {
      $targets[] = ['target_id' => $id];
    }
    return $targets;
  }

  /**
   * Check if a specific field changed between original and current.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data object.
   * @param string $field_name
   *   The name of the field to check.
   *
   * @return bool
   *   TRUE if the field changed, FALSE if not. If there is no original, there
   *   is no change.
   */
  public static function datasetFieldChanged(NodeInterface $data, string $field_name): bool {
    $original = $data->original ?? NULL;
    if ($original) {
      $old_value = Util::grabMetadata($original, $field_name);
      // @todo work out that time is different on save than on harvest so it
      // looks like it changed when it didn't (one has h:m:s the other doesn't).
      $new_value = Util::grabMetadata($data, $field_name);
      return $old_value !== $new_value;
    }
    // There is no original, so nothing changed.
    return FALSE;
  }

  /**
   * Use instead of PHP's date() to consistently get date in CMS' time zone.
   *
   * @param string $date
   *   String date to set or modify, defaults to 'now'.
   *
   * @return \DateTime
   *   Current date time .
   */
  public static function date($date = 'now'): \DateTime {
    // @todo Fix this should not be hard coded to NY time zone.
    return new \DateTime($date, new \DateTimeZone('America/New_York'));
  }

  /**
   * Get the path to the public files directory.
   *
   * @return string
   *   Path to the public files directory.
   */
  public static function getDrupalPublicFilesDirectory(): string {
    return \Drupal::service('file_system')->realpath('public://');
  }

  /**
   * Get the path to the public files directory.
   *
   * @return string
   *   Path to the public files directory.
   */
  public static function getDrupalPrivateFilesDirectory(): string {
    // @todo Fix what if private files are not enabled?
    return \Drupal::service('file_system')->realpath('private://');
  }

  /**
   * A wrapper around the static getDrupalPublicFilesDirectory function.
   *
   * @return string
   *   Path to the public files directory.
   */
  public function getDrupalPublicFilesDir() {
    return self::getDrupalPublicFilesDirectory();
  }

  /**
   * Grab metadata property from a dataset node json_metadata field.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data node.
   * @param string $property_name
   *   The name of the property to look up.
   *
   * @return mixed
   *   The content of the property, or NULL if not found.
   */
  public static function grabMetadata(NodeInterface $data, string $property_name): mixed {
    $metadata = json_decode($data->get('field_json_metadata')->getString());
    return $metadata->$property_name ?? NULL;
  }

  /**
   * Check if this data is a dataset.
   *
   * @param \Drupal\node\NodeInterface $data
   *   The data object.
   */
  public static function isDataset(NodeInterface $data): bool {
    return ($data->getType() === 'data') ? TRUE : FALSE;
  }

  /**
   * Check if a dataset is private.
   *
   * @param \Drupal\node\NodeInterface $dataset
   *   The data object.
   *
   * @return bool
   *   TRUE if the dataset is private, FALSE if not.
   */
  public static function isDatasetPrivate(NodeInterface $dataset): bool {
    $private = ['private', 'restricted', 'non-public'];
    $accessLevel = Util::grabMetadata($dataset, 'accessLevel');
    return (in_array($accessLevel, $private)) ? TRUE : FALSE;
  }

  /**
   * Make a string machine friendly.
   *
   * @param string $string
   *   String to machinize.
   *
   * @return string
   *   Machinized string.
   */
  public static function machinize($string): string {
    $new = preg_replace('/[^A-Za-z0-9 ]/', '', $string);
    $size = strlen($new);
    if ($size > 40) {
      $excess = $size - 40;
      $new = substr($new, $excess);
    }

    return $new;
  }

  /**
   * Make a string file name friendly.
   *
   * @param string $string
   *   String to file name-ize.
   *
   * @return string
   *   File name friendly string.
   */
  public static function fileNameIze($string): string {
    $new = preg_replace('/[^A-Za-z0-9_\-]/', '-', $string);
    $new = strtolower($new);

    return $new;
  }

  /**
   * Verify a directory exists, or create it.
   *
   * @param string $directory
   *   Directory.
   *
   * @return bool
   *   TRUE if it exists (or was created) and is writable. FALSE otherwise.
   */
  public static function prepareDirectory(string $directory) : bool {
    /** @var \Drupal\Core\File\FileSystem $fs */
    $fs = \Drupal::service('file_system');
    $flags = FileSystem::CREATE_DIRECTORY | FileSystem::MODIFY_PERMISSIONS;
    return $fs->prepareDirectory($directory, $flags);
  }

  /**
   * A wrapper around the static prepareDirectory function.
   *
   * @param string $directory
   *   Path to directory to prepare.
   *
   * @return bool
   *   TRUE if it exists (or was created) and is writable. FALSE otherwise.
   */
  public function prepareDir(string $directory) {
    return self::prepareDirectory($directory);
  }

  /**
   * Wraps static Term creation function so it can be mocked in phpunit.
   *
   * @param string $termName
   *   The name of the term.
   * @param string $taxonomyName
   *   The vocabulary the term belongs to.
   *
   * @return \Drupal\taxonomy\Entity\Term
   *   The metastore search api service.
   */
  public function createTaxonomyTerm(string $termName, string $taxonomyName): Term {
    return Term::create([
      'name' => $termName,
      'vid' => $taxonomyName,
    ]);
  }

}
