<?php

namespace Drupal\dkan_dataset_archiver\FileFetcher\Processor;

use Aws\S3\S3UriParser;
use Drupal\dkan_dataset_archiver\AwsS3Trait;
use FileFetcher\Processor\Remote;

/**
 * Processor for fetching files from S3.
 */
class S3 extends Remote {
  use AwsS3Trait;

  /**
   * Determine if the processor can handle the source.
   *
   * @param array $state
   *   Array of file fetcher state.
   *
   * @return bool
   *   True if the processor can handle the source, FALSE otherwise.
   */
  public function isServerCompatible(array $state): bool {
    if (substr_count($state['source'], 's3://') > 0) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Convert s3 path to a signed url for Guzzle.
   *
   * @param array $state
   *   Array of file fetcher state.
   *
   * @return array
   *   The appropriately modified state.
   */
  public function setupState(array $state): array {
    // Create a pre-signed url to the file on s3.
    $parser = new S3UriParser();
    $info = $parser->parse($state['source']);

    $s3Client = $this->getS3Client();
    $cmd = $s3Client->getCommand('GetObject', [
      'Bucket' => $info['bucket'],
      'Key' => urldecode($info['key']),
    ]);
    $request = $s3Client->createPresignedRequest($cmd, '+20 minutes');

    // Use the pre-signed url as the source.
    $state['source'] = (string) $request->getUri();

    // Copied from Remote processor.
    $state['destination'] = $this->getTemporaryFilePath($state);
    $state['temporary'] = TRUE;
    $state['total_bytes'] = 0;

    if (file_exists($state['destination'])) {
      $state['total_bytes_copied'] = $this->getFilesize($state['destination']);
    }

    return $state;
  }

  /**
   * Pre-signed urls come with a query string. Strip it for the file name.
   *
   * @param array $state
   *   Array of file fetcher state.
   *
   * @return string
   *   The directory and path.
   */
  protected function getTemporaryFileOriginalName(array $state): string {
    $path = explode("?", $state['source']);
    $file_name = basename($path[0]);
    return "{$state['temporary_directory']}/{$file_name}";
  }

}
