<?php

namespace Drupal\dkan_dataset_archiver\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Queue\RequeueException;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines 'archive_aggregation' queue worker.
 */
#[QueueWorker(
  id: 'archive_aggregation',
  title: new TranslatableMarkup('Create keyword or theme archive aggregations.'),
  cron: ['time' => 30]
)]
class ArchiveAggregation extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new ArchiveProvider Queue object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\dkan_dataset_archiver\Service\ArchiveService $archive_service
   *   The archive service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver logger channel.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ArchiveService $archive_service, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->archiveService = $archive_service;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('dkan_dataset_archiver.archive_service'),
      $container->get('logger.channel.dkan_dataset_archiver')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {

    // Attempt to process the item.
    try {
      $success = $this->archiveService->createAggregateArchive($data);
      if (!$success) {
        throw new RequeueException('Archive not created. Returning to archive_aggregation queue.');
      }

    }
    catch (\Exception $e) {
      // Log the error for debugging purposes.
      $this->logger->error('Error processing queue item: @message', ['@message' => $e->getMessage()]);
      // Re-throw the exception to put the item back in the queue.
      throw $e;
    }
  }

}
